// =============================================================================
// DriverService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { NotFoundException } from '@nestjs/common';
import { DriverService } from './driver.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  createMockDriver,
  createMockBooking,
  resetAllMocks,
} from '../../../test/setup';

describe('DriverService', () => {
  let service: DriverService;
  let prisma: typeof mockPrismaService;

  beforeEach(async () => {
    resetAllMocks();

    // Add missing prisma mocks
    mockPrismaService.driverWalletTransaction = {
      findMany: jest.fn(),
      aggregate: jest.fn(),
      count: jest.fn(),
    };
    mockPrismaService.driverVehicle = {
      findMany: jest.fn(),
      findFirst: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        DriverService,
        { provide: PrismaService, useValue: mockPrismaService },
      ],
    }).compile();

    service = module.get<DriverService>(DriverService);
    prisma = mockPrismaService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  // ===========================================================================
  // GET PROFILE
  // ===========================================================================

  describe('getProfile', () => {
    it('devrait retourner le profil du chauffeur sans mot de passe', async () => {
      const mockDriver = createMockDriver({
        vehicles: [{ id: 1, model: 'Toyota Corolla' }],
      });
      prisma.driver.findUnique.mockResolvedValue(mockDriver);

      const result = await service.getProfile(1);

      expect(result.message).toBe('Driver profile');
      expect(result.data).not.toHaveProperty('password');
      expect(result.data).not.toHaveProperty('otp');
    });

    it('devrait lever une erreur si chauffeur non trouvé', async () => {
      prisma.driver.findUnique.mockResolvedValue(null);

      await expect(service.getProfile(999)).rejects.toThrow(NotFoundException);
    });
  });

  // ===========================================================================
  // UPDATE PROFILE
  // ===========================================================================

  describe('updateProfile', () => {
    it('devrait mettre à jour le profil du chauffeur', async () => {
      const mockDriver = createMockDriver();
      prisma.driver.update.mockResolvedValue({
        ...mockDriver,
        first_name: 'Updated',
        last_name: 'Name',
      });

      const result = await service.updateProfile(1, {
        first_name: 'Updated',
        last_name: 'Name',
        email: 'updated@example.com',
      });

      expect(result.message).toBe('Profile updated successfully');
      expect(result.data.first_name).toBe('Updated');
    });
  });

  // ===========================================================================
  // UPDATE LOCATION
  // ===========================================================================

  describe('updateLocation', () => {
    it('devrait mettre à jour la position GPS du chauffeur', async () => {
      prisma.driver.update.mockResolvedValue(createMockDriver());

      const result = await service.updateLocation(1, {
        latitude: 6.1319,
        longitude: 1.2228,
      });

      expect(result.message).toBe('Location updated');
      expect(prisma.driver.update).toHaveBeenCalledWith({
        where: { id: 1 },
        data: {
          latitude: 6.1319,
          longitude: 1.2228,
        },
      });
    });
  });

  // ===========================================================================
  // UPDATE STATUS
  // ===========================================================================

  describe('updateStatus', () => {
    it('devrait mettre le chauffeur en ligne', async () => {
      prisma.driver.update.mockResolvedValue(
        createMockDriver({ is_online: 1 }),
      );

      const result = await service.updateStatus(1, { is_online: 1 });

      expect(result.message).toBe('You are now online');
      expect(result.data.is_online).toBe(1);
    });

    it('devrait mettre le chauffeur hors ligne', async () => {
      prisma.driver.update.mockResolvedValue(
        createMockDriver({ is_online: 2 }),
      );

      const result = await service.updateStatus(1, { is_online: 2 });

      expect(result.message).toBe('You are now offline');
      expect(result.data.is_online).toBe(2);
    });
  });

  // ===========================================================================
  // GET BOOKINGS
  // ===========================================================================

  describe('getBookings', () => {
    it('devrait retourner les courses du chauffeur avec pagination', async () => {
      const mockBookings = [
        createMockBooking({ id: 1, driver_id: 1 }),
        createMockBooking({ id: 2, driver_id: 1 }),
      ];
      prisma.booking.findMany.mockResolvedValue(mockBookings);
      prisma.booking.count.mockResolvedValue(2);

      const result = await service.getBookings(1, 1, 10);

      expect(result.data.bookings).toHaveLength(2);
      expect(result.data.pagination).toEqual({
        page: 1,
        limit: 10,
        total: 2,
        total_pages: 1,
      });
    });

    it('devrait calculer correctement le nombre de pages', async () => {
      prisma.booking.findMany.mockResolvedValue([]);
      prisma.booking.count.mockResolvedValue(25);

      const result = await service.getBookings(1, 1, 10);

      expect(result.data.pagination.total_pages).toBe(3);
    });
  });

  // ===========================================================================
  // GET EARNINGS
  // ===========================================================================

  describe('getEarnings', () => {
    it('devrait retourner les gains totaux et du jour', async () => {
      prisma.booking.aggregate
        .mockResolvedValueOnce({
          _sum: { final_amount: 150000 },
          _count: 30,
        })
        .mockResolvedValueOnce({
          _sum: { final_amount: 15000 },
          _count: 5,
        });

      const result = await service.getEarnings(1);

      expect(result.data.total_earnings).toBe(150000);
      expect(result.data.total_rides).toBe(30);
      expect(result.data.today_earnings).toBe(15000);
      expect(result.data.today_rides).toBe(5);
    });

    it('devrait filtrer par dates', async () => {
      prisma.booking.aggregate
        .mockResolvedValueOnce({
          _sum: { final_amount: 50000 },
          _count: 10,
        })
        .mockResolvedValueOnce({
          _sum: { final_amount: 5000 },
          _count: 2,
        });

      const result = await service.getEarnings(
        1,
        '2024-01-01',
        '2024-01-31',
      );

      expect(result.data.total_earnings).toBe(50000);
    });

    it('devrait retourner 0 si pas de courses', async () => {
      prisma.booking.aggregate.mockResolvedValue({
        _sum: { final_amount: null },
        _count: 0,
      });

      const result = await service.getEarnings(1);

      expect(result.data.total_earnings).toBe(0);
      expect(result.data.total_rides).toBe(0);
    });
  });

  // ===========================================================================
  // GET WALLET
  // ===========================================================================

  describe('getWallet', () => {
    it('devrait retourner le solde et les transactions', async () => {
      const mockTransactions = [
        { id: 1, amount: 10000, type: 'credit', created_at: new Date() },
        { id: 2, amount: -5000, type: 'debit', created_at: new Date() },
      ];

      prisma.driverWalletTransaction.aggregate.mockResolvedValue({
        _sum: { amount: 5000 },
      });
      prisma.driverWalletTransaction.findMany.mockResolvedValue(mockTransactions);
      prisma.driverWalletTransaction.count.mockResolvedValue(2);

      const result = await service.getWallet(1, 1, 10);

      expect(result.data.balance).toBe(5000);
      expect(result.data.transactions).toHaveLength(2);
    });

    it('devrait retourner 0 si pas de transactions', async () => {
      prisma.driverWalletTransaction.aggregate.mockResolvedValue({
        _sum: { amount: null },
      });
      prisma.driverWalletTransaction.findMany.mockResolvedValue([]);
      prisma.driverWalletTransaction.count.mockResolvedValue(0);

      const result = await service.getWallet(1, 1, 10);

      expect(result.data.balance).toBe(0);
    });
  });

  // ===========================================================================
  // GET VEHICLES
  // ===========================================================================

  describe('getVehicles', () => {
    it('devrait retourner les véhicules du chauffeur', async () => {
      const mockVehicles = [
        { id: 1, model: 'Toyota Corolla', is_default: 1 },
        { id: 2, model: 'Honda Civic', is_default: 0 },
      ];
      prisma.driverVehicle.findMany.mockResolvedValue(mockVehicles);

      const result = await service.getVehicles(1);

      expect(result.data).toHaveLength(2);
      expect(result.data[0].is_default).toBe(1);
    });

    it('devrait retourner une liste vide si pas de véhicules', async () => {
      prisma.driverVehicle.findMany.mockResolvedValue([]);

      const result = await service.getVehicles(1);

      expect(result.data).toHaveLength(0);
    });
  });
});
