import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

// Document types
export enum DocumentType {
  DRIVERS_LICENSE = 'drivers_license',
  VEHICLE_REGISTRATION = 'vehicle_registration',
  INSURANCE = 'insurance',
  BACKGROUND_CHECK = 'background_check',
  PROFILE_PHOTO = 'profile_photo',
  VEHICLE_PHOTO = 'vehicle_photo',
  ID_CARD = 'id_card',
  PROOF_OF_ADDRESS = 'proof_of_address',
}

// Document status
export enum DocumentStatus {
  PENDING = 'pending',
  APPROVED = 'approved',
  REJECTED = 'rejected',
  EXPIRED = 'expired',
}

interface DriverDocument {
  id: number;
  driverId: number;
  type: DocumentType;
  name: string;
  fileUrl: string;
  status: DocumentStatus;
  expiryDate?: Date;
  rejectionReason?: string;
  verifiedAt?: Date;
  verifiedBy?: number;
  createdAt: Date;
}

interface DocumentRequirement {
  type: DocumentType;
  name: string;
  description: string;
  isRequired: boolean;
  hasExpiry: boolean;
  allowedFormats: string[];
  maxSizeMB: number;
}

@Injectable()
export class DocumentService {
  private readonly logger = new Logger(DocumentService.name);

  // Document requirements
  private readonly DOCUMENT_REQUIREMENTS: DocumentRequirement[] = [
    {
      type: DocumentType.DRIVERS_LICENSE,
      name: 'Permis de conduire',
      description: 'Photo recto-verso de votre permis de conduire',
      isRequired: true,
      hasExpiry: true,
      allowedFormats: ['jpg', 'jpeg', 'png', 'pdf'],
      maxSizeMB: 5,
    },
    {
      type: DocumentType.VEHICLE_REGISTRATION,
      name: 'Carte grise',
      description: 'Photo de la carte grise du vehicule',
      isRequired: true,
      hasExpiry: true,
      allowedFormats: ['jpg', 'jpeg', 'png', 'pdf'],
      maxSizeMB: 5,
    },
    {
      type: DocumentType.INSURANCE,
      name: 'Assurance',
      description: 'Attestation d\'assurance du vehicule',
      isRequired: true,
      hasExpiry: true,
      allowedFormats: ['jpg', 'jpeg', 'png', 'pdf'],
      maxSizeMB: 5,
    },
    {
      type: DocumentType.ID_CARD,
      name: 'Carte d\'identite',
      description: 'Photo recto-verso de votre piece d\'identite',
      isRequired: true,
      hasExpiry: true,
      allowedFormats: ['jpg', 'jpeg', 'png', 'pdf'],
      maxSizeMB: 5,
    },
    {
      type: DocumentType.PROFILE_PHOTO,
      name: 'Photo de profil',
      description: 'Photo claire de votre visage',
      isRequired: true,
      hasExpiry: false,
      allowedFormats: ['jpg', 'jpeg', 'png'],
      maxSizeMB: 2,
    },
    {
      type: DocumentType.VEHICLE_PHOTO,
      name: 'Photo du vehicule',
      description: 'Photo de votre vehicule (exterieur)',
      isRequired: true,
      hasExpiry: false,
      allowedFormats: ['jpg', 'jpeg', 'png'],
      maxSizeMB: 5,
    },
    {
      type: DocumentType.BACKGROUND_CHECK,
      name: 'Casier judiciaire',
      description: 'Extrait de casier judiciaire (moins de 3 mois)',
      isRequired: false,
      hasExpiry: true,
      allowedFormats: ['jpg', 'jpeg', 'png', 'pdf'],
      maxSizeMB: 5,
    },
  ];

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  /**
   * Get document requirements for a merchant
   */
  async getDocumentRequirements(merchantId: number): Promise<DocumentRequirement[]> {
    // In a real implementation, this could be customized per merchant
    return this.DOCUMENT_REQUIREMENTS;
  }

  /**
   * Upload a document
   */
  async uploadDocument(
    driverId: number,
    merchantId: number,
    type: DocumentType,
    fileUrl: string,
    expiryDate?: Date,
  ): Promise<DriverDocument> {
    // Check if document type exists
    const requirement = this.DOCUMENT_REQUIREMENTS.find((r) => r.type === type);
    if (!requirement) {
      throw new BadRequestException('Type de document invalide');
    }

    // Check for existing pending/approved document of same type
    const existing = await this.prisma.driverDocument.findFirst({
      where: {
        driver_id: driverId,
        type,
        status: { in: ['pending', 'approved'] },
      },
    });

    if (existing) {
      // Update existing document
      const updated = await this.prisma.driverDocument.update({
        where: { id: existing.id },
        data: {
          file_url: fileUrl,
          expiry_date: expiryDate,
          status: 'pending',
          rejection_reason: null,
          updated_at: new Date(),
        },
      });

      this.logger.log(`Driver #${driverId} updated document ${type}`);
      return this.formatDocument(updated);
    }

    // Create new document
    const document = await this.prisma.driverDocument.create({
      data: {
        driver_id: driverId,
        merchant_id: merchantId,
        type,
        name: requirement.name,
        file_url: fileUrl,
        expiry_date: expiryDate,
        status: 'pending',
        created_at: new Date(),
      },
    });

    this.logger.log(`Driver #${driverId} uploaded document ${type}`);
    return this.formatDocument(document);
  }

  /**
   * Get driver documents
   */
  async getDriverDocuments(driverId: number): Promise<{
    documents: DriverDocument[];
    requirements: DocumentRequirement[];
    completionPercentage: number;
    missingRequired: DocumentType[];
  }> {
    const documents = await this.prisma.driverDocument.findMany({
      where: { driver_id: driverId },
      orderBy: { created_at: 'desc' },
    });

    // Get unique documents by type (latest status)
    const documentMap = new Map<string, any>();
    for (const doc of documents) {
      if (!documentMap.has(doc.type)) {
        documentMap.set(doc.type, doc);
      }
    }

    const uniqueDocuments = Array.from(documentMap.values());

    // Calculate completion
    const requiredTypes = this.DOCUMENT_REQUIREMENTS
      .filter((r) => r.isRequired)
      .map((r) => r.type);

    const approvedTypes = uniqueDocuments
      .filter((d) => d.status === 'approved')
      .map((d) => d.type);

    const missingRequired = requiredTypes.filter(
      (type) => !approvedTypes.includes(type),
    );

    const completionPercentage = requiredTypes.length > 0
      ? Math.round((approvedTypes.filter((t) => requiredTypes.includes(t)).length / requiredTypes.length) * 100)
      : 100;

    return {
      documents: uniqueDocuments.map((d) => this.formatDocument(d)),
      requirements: this.DOCUMENT_REQUIREMENTS,
      completionPercentage,
      missingRequired,
    };
  }

  /**
   * Get document by ID
   */
  async getDocument(documentId: number): Promise<DriverDocument | null> {
    const document = await this.prisma.driverDocument.findUnique({
      where: { id: documentId },
    });

    return document ? this.formatDocument(document) : null;
  }

  /**
   * Admin: Approve document
   */
  async approveDocument(
    documentId: number,
    adminId: number,
  ): Promise<DriverDocument> {
    const document = await this.prisma.driverDocument.findUnique({
      where: { id: documentId },
    });

    if (!document) {
      throw new NotFoundException('Document non trouve');
    }

    if (document.status !== 'pending') {
      throw new BadRequestException('Ce document n\'est pas en attente');
    }

    const updated = await this.prisma.driverDocument.update({
      where: { id: documentId },
      data: {
        status: 'approved',
        verified_at: new Date(),
        verified_by: adminId,
      },
    });

    // Check if all required documents are approved
    await this.checkDriverDocumentCompletion(document.driver_id);

    // Notify driver
    await this.notificationService.sendToDriver(document.driver_id, {
      title: 'Document approuve',
      body: `Votre ${document.name} a ete approuve`,
      data: { type: 'document_approved', document_id: String(documentId) },
    });

    this.logger.log(`Document #${documentId} approved by admin #${adminId}`);
    return this.formatDocument(updated);
  }

  /**
   * Admin: Reject document
   */
  async rejectDocument(
    documentId: number,
    adminId: number,
    reason: string,
  ): Promise<DriverDocument> {
    const document = await this.prisma.driverDocument.findUnique({
      where: { id: documentId },
    });

    if (!document) {
      throw new NotFoundException('Document non trouve');
    }

    const updated = await this.prisma.driverDocument.update({
      where: { id: documentId },
      data: {
        status: 'rejected',
        rejection_reason: reason,
        verified_at: new Date(),
        verified_by: adminId,
      },
    });

    // Notify driver
    await this.notificationService.sendToDriver(document.driver_id, {
      title: 'Document rejete',
      body: `Votre ${document.name} a ete rejete: ${reason}`,
      data: { type: 'document_rejected', document_id: String(documentId) },
    });

    this.logger.log(`Document #${documentId} rejected by admin #${adminId}`);
    return this.formatDocument(updated);
  }

  /**
   * Admin: Get pending documents
   */
  async getPendingDocuments(
    merchantId: number,
    options?: { page?: number; limit?: number; type?: DocumentType },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = {
      merchant_id: merchantId,
      status: 'pending',
    };

    if (options?.type) {
      where.type = options.type;
    }

    const [documents, total] = await Promise.all([
      this.prisma.driverDocument.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'asc' },
        include: {
          driver: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              email: true,
              mobile: true,
            },
          },
        },
      }),
      this.prisma.driverDocument.count({ where }),
    ]);

    return {
      data: documents.map((d) => ({
        ...this.formatDocument(d),
        driver: d.driver,
      })),
      total,
    };
  }

  /**
   * Check for expiring documents
   */
  async checkExpiringDocuments(daysAhead: number = 30): Promise<void> {
    const expiryThreshold = new Date();
    expiryThreshold.setDate(expiryThreshold.getDate() + daysAhead);

    const expiringDocuments = await this.prisma.driverDocument.findMany({
      where: {
        status: 'approved',
        expiry_date: {
          lte: expiryThreshold,
          gte: new Date(),
        },
      },
      include: {
        driver: { select: { id: true, first_name: true } },
      },
    });

    for (const doc of expiringDocuments) {
      const daysUntilExpiry = Math.ceil(
        (doc.expiry_date.getTime() - Date.now()) / (1000 * 60 * 60 * 24),
      );

      await this.notificationService.sendToDriver(doc.driver_id, {
        title: 'Document bientot expire',
        body: `Votre ${doc.name} expire dans ${daysUntilExpiry} jours`,
        data: { type: 'document_expiring', document_id: String(doc.id) },
      });
    }

    this.logger.log(`Sent expiry notifications for ${expiringDocuments.length} documents`);
  }

  /**
   * Mark expired documents
   */
  async markExpiredDocuments(): Promise<number> {
    const result = await this.prisma.driverDocument.updateMany({
      where: {
        status: 'approved',
        expiry_date: { lt: new Date() },
      },
      data: { status: 'expired' },
    });

    if (result.count > 0) {
      this.logger.log(`Marked ${result.count} documents as expired`);
    }

    return result.count;
  }

  /**
   * Check if driver has all required documents approved
   */
  private async checkDriverDocumentCompletion(driverId: number): Promise<void> {
    const { completionPercentage, missingRequired } = await this.getDriverDocuments(driverId);

    if (completionPercentage === 100 && missingRequired.length === 0) {
      // All required documents approved - update driver status
      const driver = await this.prisma.driver.findUnique({
        where: { id: driverId },
      });

      if (driver && driver.driver_status === 0) {
        // Status 0 = pending approval
        await this.prisma.driver.update({
          where: { id: driverId },
          data: { driver_status: 1 }, // 1 = approved
        });

        await this.notificationService.sendToDriver(driverId, {
          title: 'Compte active!',
          body: 'Tous vos documents sont approuves. Vous pouvez maintenant recevoir des courses.',
          data: { type: 'account_activated' },
        });

        this.logger.log(`Driver #${driverId} account activated - all documents approved`);
      }
    }
  }

  /**
   * Format document for response
   */
  private formatDocument(document: any): DriverDocument {
    return {
      id: document.id,
      driverId: document.driver_id,
      type: document.type as DocumentType,
      name: document.name,
      fileUrl: document.file_url,
      status: document.status as DocumentStatus,
      expiryDate: document.expiry_date,
      rejectionReason: document.rejection_reason,
      verifiedAt: document.verified_at,
      verifiedBy: document.verified_by,
      createdAt: document.created_at,
    };
  }
}
