import {
  IsString,
  IsNumber,
  IsOptional,
  IsBoolean,
  IsEnum,
  IsDateString,
  Min,
  MaxLength,
  IsPhoneNumber,
} from 'class-validator';
import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';

export class CreateDeliveryDto {
  // Pickup details
  @ApiProperty({ description: 'Pickup address' })
  @IsString()
  @MaxLength(500)
  pickupAddress: string;

  @ApiProperty({ description: 'Pickup latitude' })
  @IsNumber()
  pickupLatitude: number;

  @ApiProperty({ description: 'Pickup longitude' })
  @IsNumber()
  pickupLongitude: number;

  @ApiProperty({ description: 'Pickup contact name' })
  @IsString()
  @MaxLength(100)
  pickupContactName: string;

  @ApiProperty({ description: 'Pickup contact phone' })
  @IsString()
  pickupContactPhone: string;

  @ApiPropertyOptional({ description: 'Pickup instructions' })
  @IsOptional()
  @IsString()
  @MaxLength(500)
  pickupInstructions?: string;

  // Drop details
  @ApiProperty({ description: 'Drop address' })
  @IsString()
  @MaxLength(500)
  dropAddress: string;

  @ApiProperty({ description: 'Drop latitude' })
  @IsNumber()
  dropLatitude: number;

  @ApiProperty({ description: 'Drop longitude' })
  @IsNumber()
  dropLongitude: number;

  @ApiProperty({ description: 'Drop contact name' })
  @IsString()
  @MaxLength(100)
  dropContactName: string;

  @ApiProperty({ description: 'Drop contact phone' })
  @IsString()
  dropContactPhone: string;

  @ApiPropertyOptional({ description: 'Drop instructions' })
  @IsOptional()
  @IsString()
  @MaxLength(500)
  dropInstructions?: string;

  // Package details
  @ApiProperty({ description: 'Package type', example: 'document' })
  @IsString()
  @MaxLength(50)
  packageType: string;

  @ApiPropertyOptional({ description: 'Package weight in kg' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  packageWeight?: number;

  @ApiPropertyOptional({
    description: 'Package size',
    enum: ['small', 'medium', 'large', 'extra_large'],
  })
  @IsOptional()
  @IsEnum(['small', 'medium', 'large', 'extra_large'])
  packageSize?: 'small' | 'medium' | 'large' | 'extra_large';

  @ApiPropertyOptional({ description: 'Package description' })
  @IsOptional()
  @IsString()
  @MaxLength(500)
  packageDescription?: string;

  @ApiPropertyOptional({ description: 'Declared value for insurance' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  declaredValue?: number;

  @ApiPropertyOptional({ description: 'Is package fragile' })
  @IsOptional()
  @IsBoolean()
  isFragile?: boolean;

  @ApiPropertyOptional({ description: 'Requires signature on delivery' })
  @IsOptional()
  @IsBoolean()
  requiresSignature?: boolean;

  // Options
  @ApiPropertyOptional({ description: 'Vehicle type ID' })
  @IsOptional()
  @IsNumber()
  vehicleTypeId?: number;

  @ApiPropertyOptional({ description: 'Scheduled pickup time (ISO 8601)' })
  @IsOptional()
  @IsDateString()
  scheduledPickupTime?: string;

  @ApiProperty({ description: 'Payment type', enum: ['cash', 'card', 'wallet'] })
  @IsEnum(['cash', 'card', 'wallet'])
  paymentType: 'cash' | 'card' | 'wallet';

  @ApiPropertyOptional({ description: 'Promo code' })
  @IsOptional()
  @IsString()
  promoCode?: string;
}
