import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron } from '@nestjs/schedule';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class ReportTasksService implements OnModuleInit {
  private readonly logger = new Logger(ReportTasksService.name);

  constructor(private prisma: PrismaService) {}

  onModuleInit() {
    this.logger.log('Report Tasks initialized');
  }

  /**
   * Generate daily statistics
   * Runs daily at midnight
   */
  @Cron('0 0 * * *')
  async generateDailyStats() {
    this.logger.debug('Running: generateDailyStats');

    try {
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      yesterday.setHours(0, 0, 0, 0);

      const today = new Date();
      today.setHours(0, 0, 0, 0);

      // Get all merchants
      const merchants = await this.prisma.merchant.findMany({
        select: { id: true },
      });

      for (const merchant of merchants) {
        // Booking stats
        const bookingStats = await this.prisma.booking.groupBy({
          by: ['status'],
          where: {
            merchant_id: merchant.id,
            created_at: { gte: yesterday, lt: today },
          },
          _count: { id: true },
          _sum: { total_fare: true },
        });

        // New users
        const newUsers = await this.prisma.user.count({
          where: {
            merchant_id: merchant.id,
            created_at: { gte: yesterday, lt: today },
          },
        });

        // New drivers
        const newDrivers = await this.prisma.driver.count({
          where: {
            merchant_id: merchant.id,
            created_at: { gte: yesterday, lt: today },
          },
        });

        // Active drivers
        const activeDrivers = await this.prisma.driver.count({
          where: {
            merchant_id: merchant.id,
            bookings: {
              some: {
                status: 'completed',
                ride_ended_at: { gte: yesterday, lt: today },
              },
            },
          },
        });

        // Store stats
        await this.prisma.dailyStats.create({
          data: {
            merchant_id: merchant.id,
            date: yesterday,
            total_bookings: bookingStats.reduce((sum, s) => sum + s._count.id, 0),
            completed_bookings: bookingStats.find(s => s.status === 'completed')?._count.id || 0,
            cancelled_bookings: bookingStats.find(s => s.status === 'cancelled')?._count.id || 0,
            total_revenue: bookingStats
              .filter(s => s.status === 'completed')
              .reduce((sum, s) => sum + (s._sum.total_fare || 0), 0),
            new_users: newUsers,
            new_drivers: newDrivers,
            active_drivers: activeDrivers,
            created_at: new Date(),
          },
        });
      }

      this.logger.log(`Generated daily stats for ${merchants.length} merchants`);
    } catch (error) {
      this.logger.error(`generateDailyStats failed: ${error.message}`);
    }
  }

  /**
   * Generate weekly report
   * Runs every Monday at 1 AM
   */
  @Cron('0 1 * * 1')
  async generateWeeklyReport() {
    this.logger.debug('Running: generateWeeklyReport');

    try {
      const lastWeekStart = new Date();
      lastWeekStart.setDate(lastWeekStart.getDate() - 7);
      lastWeekStart.setHours(0, 0, 0, 0);

      const lastWeekEnd = new Date();
      lastWeekEnd.setHours(0, 0, 0, 0);

      const merchants = await this.prisma.merchant.findMany({
        where: { weekly_report_enabled: true },
      });

      for (const merchant of merchants) {
        // Aggregate weekly stats
        const weeklyStats = await this.prisma.dailyStats.aggregate({
          where: {
            merchant_id: merchant.id,
            date: { gte: lastWeekStart, lt: lastWeekEnd },
          },
          _sum: {
            total_bookings: true,
            completed_bookings: true,
            cancelled_bookings: true,
            total_revenue: true,
            new_users: true,
            new_drivers: true,
          },
          _avg: {
            active_drivers: true,
          },
        });

        // Store weekly report
        await this.prisma.weeklyReport.create({
          data: {
            merchant_id: merchant.id,
            week_start: lastWeekStart,
            week_end: lastWeekEnd,
            total_bookings: weeklyStats._sum.total_bookings || 0,
            completed_bookings: weeklyStats._sum.completed_bookings || 0,
            cancelled_bookings: weeklyStats._sum.cancelled_bookings || 0,
            total_revenue: weeklyStats._sum.total_revenue || 0,
            new_users: weeklyStats._sum.new_users || 0,
            new_drivers: weeklyStats._sum.new_drivers || 0,
            avg_active_drivers: weeklyStats._avg.active_drivers || 0,
            created_at: new Date(),
          },
        });

        // TODO: Send email report
        this.logger.debug(`Generated weekly report for merchant ${merchant.id}`);
      }

      this.logger.log(`Generated weekly reports for ${merchants.length} merchants`);
    } catch (error) {
      this.logger.error(`generateWeeklyReport failed: ${error.message}`);
    }
  }

  /**
   * Generate monthly report
   * Runs on the 1st of each month at 2 AM
   */
  @Cron('0 2 1 * *')
  async generateMonthlyReport() {
    this.logger.debug('Running: generateMonthlyReport');

    try {
      const lastMonthStart = new Date();
      lastMonthStart.setMonth(lastMonthStart.getMonth() - 1);
      lastMonthStart.setDate(1);
      lastMonthStart.setHours(0, 0, 0, 0);

      const lastMonthEnd = new Date();
      lastMonthEnd.setDate(1);
      lastMonthEnd.setHours(0, 0, 0, 0);

      const merchants = await this.prisma.merchant.findMany({
        where: { monthly_report_enabled: true },
      });

      for (const merchant of merchants) {
        // Aggregate monthly stats
        const monthlyStats = await this.prisma.dailyStats.aggregate({
          where: {
            merchant_id: merchant.id,
            date: { gte: lastMonthStart, lt: lastMonthEnd },
          },
          _sum: {
            total_bookings: true,
            completed_bookings: true,
            cancelled_bookings: true,
            total_revenue: true,
            new_users: true,
            new_drivers: true,
          },
        });

        // Top drivers
        const topDrivers = await this.prisma.booking.groupBy({
          by: ['driver_id'],
          where: {
            merchant_id: merchant.id,
            status: 'completed',
            ride_ended_at: { gte: lastMonthStart, lt: lastMonthEnd },
            driver_id: { not: null },
          },
          _count: { id: true },
          _sum: { driver_earning: true },
          orderBy: { _count: { id: 'desc' } },
          take: 10,
        });

        // Store monthly report
        await this.prisma.monthlyReport.create({
          data: {
            merchant_id: merchant.id,
            month: lastMonthStart.getMonth() + 1,
            year: lastMonthStart.getFullYear(),
            total_bookings: monthlyStats._sum.total_bookings || 0,
            completed_bookings: monthlyStats._sum.completed_bookings || 0,
            cancelled_bookings: monthlyStats._sum.cancelled_bookings || 0,
            total_revenue: monthlyStats._sum.total_revenue || 0,
            new_users: monthlyStats._sum.new_users || 0,
            new_drivers: monthlyStats._sum.new_drivers || 0,
            top_drivers: JSON.stringify(topDrivers),
            created_at: new Date(),
          },
        });

        // TODO: Generate PDF and send email
        this.logger.debug(`Generated monthly report for merchant ${merchant.id}`);
      }

      this.logger.log(`Generated monthly reports for ${merchants.length} merchants`);
    } catch (error) {
      this.logger.error(`generateMonthlyReport failed: ${error.message}`);
    }
  }

  /**
   * Calculate surge pricing zones
   * Runs every 5 minutes
   */
  @Cron('*/5 * * * *')
  async calculateSurgePricing() {
    this.logger.debug('Running: calculateSurgePricing');

    try {
      // Get demand vs supply in each zone
      const zones = await this.prisma.zone.findMany({
        where: { surge_enabled: true },
      });

      for (const zone of zones) {
        // Count pending bookings in zone
        const pendingBookings = await this.prisma.booking.count({
          where: {
            status: 'pending',
            pickup_lat: { gte: zone.min_lat, lte: zone.max_lat },
            pickup_lng: { gte: zone.min_lng, lte: zone.max_lng },
          },
        });

        // Count available drivers in zone
        const availableDrivers = await this.prisma.driver.count({
          where: {
            is_online: true,
            is_available: true,
            current_lat: { gte: zone.min_lat, lte: zone.max_lat },
            current_lng: { gte: zone.min_lng, lte: zone.max_lng },
          },
        });

        // Calculate surge multiplier
        let surgeMultiplier = 1.0;
        if (availableDrivers > 0) {
          const ratio = pendingBookings / availableDrivers;
          if (ratio > 2) surgeMultiplier = 2.0;
          else if (ratio > 1.5) surgeMultiplier = 1.5;
          else if (ratio > 1) surgeMultiplier = 1.25;
        } else if (pendingBookings > 0) {
          surgeMultiplier = 2.5; // No drivers available
        }

        // Update zone surge
        if (zone.surge_multiplier !== surgeMultiplier) {
          await this.prisma.zone.update({
            where: { id: zone.id },
            data: { surge_multiplier: surgeMultiplier },
          });
          this.logger.debug(`Zone ${zone.name} surge: ${surgeMultiplier}x`);
        }
      }
    } catch (error) {
      this.logger.error(`calculateSurgePricing failed: ${error.message}`);
    }
  }

  /**
   * Generate franchise revenue reports
   * Runs daily at 5 AM
   */
  @Cron('0 5 * * *')
  async generateFranchiseReports() {
    this.logger.debug('Running: generateFranchiseReports');

    try {
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      yesterday.setHours(0, 0, 0, 0);

      const today = new Date();
      today.setHours(0, 0, 0, 0);

      const franchises = await this.prisma.franchise.findMany({
        where: { is_active: true },
      });

      for (const franchise of franchises) {
        const revenue = await this.prisma.booking.aggregate({
          where: {
            franchise_id: franchise.id,
            status: 'completed',
            ride_ended_at: { gte: yesterday, lt: today },
          },
          _sum: { total_fare: true },
          _count: { id: true },
        });

        const commission = (revenue._sum.total_fare || 0) * (franchise.commission_rate / 100);

        await this.prisma.franchiseRevenue.create({
          data: {
            franchise_id: franchise.id,
            date: yesterday,
            total_bookings: revenue._count.id,
            total_revenue: revenue._sum.total_fare || 0,
            commission_amount: commission,
            created_at: new Date(),
          },
        });
      }

      this.logger.log(`Generated reports for ${franchises.length} franchises`);
    } catch (error) {
      this.logger.error(`generateFranchiseReports failed: ${error.message}`);
    }
  }
}
