import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class NotificationTasksService implements OnModuleInit {
  private readonly logger = new Logger(NotificationTasksService.name);

  constructor(private prisma: PrismaService) {}

  onModuleInit() {
    this.logger.log('Notification Tasks initialized');
  }

  /**
   * Send pending push notifications
   * Runs every minute
   */
  @Cron(CronExpression.EVERY_MINUTE)
  async sendPendingNotifications() {
    this.logger.debug('Running: sendPendingNotifications');

    try {
      const pendingNotifications = await this.prisma.notification.findMany({
        where: {
          status: 'pending',
          scheduled_at: { lte: new Date() },
        },
        take: 100,
      });

      for (const notification of pendingNotifications) {
        try {
          // TODO: Send via FCM/OneSignal
          await this.prisma.notification.update({
            where: { id: notification.id },
            data: {
              status: 'sent',
              sent_at: new Date(),
            },
          });
        } catch (error) {
          await this.prisma.notification.update({
            where: { id: notification.id },
            data: {
              status: 'failed',
              error: error.message,
            },
          });
        }
      }

      if (pendingNotifications.length > 0) {
        this.logger.debug(`Processed ${pendingNotifications.length} notifications`);
      }
    } catch (error) {
      this.logger.error(`sendPendingNotifications failed: ${error.message}`);
    }
  }

  /**
   * Send daily summary to merchants
   * Runs daily at 7 AM
   */
  @Cron('0 7 * * *')
  async sendMerchantDailySummary() {
    this.logger.debug('Running: sendMerchantDailySummary');

    try {
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      yesterday.setHours(0, 0, 0, 0);

      const today = new Date();
      today.setHours(0, 0, 0, 0);

      const merchants = await this.prisma.merchant.findMany({
        where: {
          daily_summary_enabled: true,
          email: { not: null },
        },
      });

      for (const merchant of merchants) {
        const stats = await this.prisma.booking.aggregate({
          where: {
            merchant_id: merchant.id,
            created_at: { gte: yesterday, lt: today },
          },
          _count: { id: true },
          _sum: { total_fare: true },
        });

        // TODO: Send email with stats
        this.logger.debug(`Daily summary for merchant ${merchant.id}: ${stats._count.id} bookings`);
      }
    } catch (error) {
      this.logger.error(`sendMerchantDailySummary failed: ${error.message}`);
    }
  }

  /**
   * Send re-engagement notifications to inactive users
   * Runs daily at 10 AM
   */
  @Cron('0 10 * * *')
  async sendReEngagementNotifications() {
    this.logger.debug('Running: sendReEngagementNotifications');

    try {
      const inactiveDate = new Date();
      inactiveDate.setDate(inactiveDate.getDate() - 14); // 14 days inactive

      const inactiveUsers = await this.prisma.user.findMany({
        where: {
          last_booking_at: { lt: inactiveDate },
          push_enabled: true,
          re_engagement_sent_at: null,
        },
        take: 100,
      });

      for (const user of inactiveUsers) {
        // TODO: Send push notification
        await this.prisma.user.update({
          where: { id: user.id },
          data: { re_engagement_sent_at: new Date() },
        });
      }

      this.logger.log(`Sent re-engagement to ${inactiveUsers.length} users`);
    } catch (error) {
      this.logger.error(`sendReEngagementNotifications failed: ${error.message}`);
    }
  }

  /**
   * Send birthday notifications
   * Runs daily at 9 AM
   */
  @Cron('0 9 * * *')
  async sendBirthdayNotifications() {
    this.logger.debug('Running: sendBirthdayNotifications');

    try {
      const today = new Date();
      const month = today.getMonth() + 1;
      const day = today.getDate();

      // Find users with birthday today
      const birthdayUsers = await this.prisma.$queryRaw`
        SELECT * FROM users
        WHERE MONTH(date_of_birth) = ${month}
        AND DAY(date_of_birth) = ${day}
        AND push_enabled = true
      `;

      // TODO: Send birthday notifications
      this.logger.log(`Found ${(birthdayUsers as any[]).length} users with birthday today`);
    } catch (error) {
      this.logger.error(`sendBirthdayNotifications failed: ${error.message}`);
    }
  }

  /**
   * Clean old notifications
   * Runs weekly on Sunday at 3 AM
   */
  @Cron('0 3 * * 0')
  async cleanOldNotifications() {
    this.logger.debug('Running: cleanOldNotifications');

    try {
      const thirtyDaysAgo = new Date();
      thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

      const result = await this.prisma.notification.deleteMany({
        where: {
          created_at: { lt: thirtyDaysAgo },
          status: { in: ['sent', 'read', 'failed'] },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} old notifications`);
      }
    } catch (error) {
      this.logger.error(`cleanOldNotifications failed: ${error.message}`);
    }
  }

  /**
   * Send weather alerts to drivers
   * Runs every 6 hours
   */
  @Cron('0 */6 * * *')
  async sendWeatherAlerts() {
    this.logger.debug('Running: sendWeatherAlerts');

    // TODO: Integrate weather API and send alerts for bad weather
    this.logger.debug('Weather alerts check completed');
  }
}
