import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class DriverTasksService implements OnModuleInit {
  private readonly logger = new Logger(DriverTasksService.name);

  constructor(private prisma: PrismaService) {}

  onModuleInit() {
    this.logger.log('Driver Tasks initialized');
  }

  /**
   * Auto-offline inactive drivers
   * Runs every 10 minutes
   */
  @Cron(CronExpression.EVERY_10_MINUTES)
  async autoOfflineInactiveDrivers() {
    this.logger.debug('Running: autoOfflineInactiveDrivers');

    try {
      const inactiveTime = new Date(Date.now() - 30 * 60 * 1000); // 30 minutes

      const result = await this.prisma.driver.updateMany({
        where: {
          is_online: true,
          last_location_update: { lt: inactiveTime },
          // Don't offline drivers currently on a booking
          NOT: {
            bookings: {
              some: {
                status: { in: ['accepted', 'arrived', 'in_progress'] },
              },
            },
          },
        },
        data: {
          is_online: false,
          went_offline_at: new Date(),
        },
      });

      if (result.count > 0) {
        this.logger.log(`Auto-offlined ${result.count} inactive drivers`);
      }
    } catch (error) {
      this.logger.error(`autoOfflineInactiveDrivers failed: ${error.message}`);
    }
  }

  /**
   * Check document expiration and notify drivers
   * Runs daily at 8 AM
   */
  @Cron('0 8 * * *')
  async checkDocumentExpiration() {
    this.logger.debug('Running: checkDocumentExpiration');

    try {
      const warningDate = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000); // 7 days
      const expiredDate = new Date();

      // Find expiring documents
      const expiringDocs = await this.prisma.driverDocument.findMany({
        where: {
          status: 'approved',
          expiry_date: {
            gte: expiredDate,
            lte: warningDate,
          },
          expiry_notified: false,
        },
        include: {
          driver: { select: { id: true, name: true, phone: true, email: true } },
        },
      });

      for (const doc of expiringDocs) {
        // TODO: Send notification to driver
        await this.prisma.driverDocument.update({
          where: { id: doc.id },
          data: { expiry_notified: true },
        });
        this.logger.log(`Expiry notification sent to driver ${doc.driver.name} for ${doc.document_type}`);
      }

      // Mark expired documents
      const result = await this.prisma.driverDocument.updateMany({
        where: {
          status: 'approved',
          expiry_date: { lt: expiredDate },
        },
        data: {
          status: 'expired',
        },
      });

      if (result.count > 0) {
        this.logger.log(`Marked ${result.count} documents as expired`);
      }

      // Deactivate drivers with expired mandatory documents
      const driversWithExpired = await this.prisma.driver.findMany({
        where: {
          is_active: true,
          documents: {
            some: {
              status: 'expired',
              is_mandatory: true,
            },
          },
        },
      });

      for (const driver of driversWithExpired) {
        await this.prisma.driver.update({
          where: { id: driver.id },
          data: {
            is_active: false,
            deactivation_reason: 'expired_documents',
          },
        });
        this.logger.warn(`Deactivated driver ${driver.id} due to expired documents`);
      }
    } catch (error) {
      this.logger.error(`checkDocumentExpiration failed: ${error.message}`);
    }
  }

  /**
   * Calculate daily driver earnings
   * Runs daily at midnight
   */
  @Cron('0 0 * * *')
  async calculateDailyEarnings() {
    this.logger.debug('Running: calculateDailyEarnings');

    try {
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      yesterday.setHours(0, 0, 0, 0);

      const today = new Date();
      today.setHours(0, 0, 0, 0);

      // Get all completed bookings from yesterday
      const earnings = await this.prisma.booking.groupBy({
        by: ['driver_id'],
        where: {
          status: 'completed',
          ride_ended_at: {
            gte: yesterday,
            lt: today,
          },
          driver_id: { not: null },
        },
        _sum: {
          driver_earning: true,
          tip_amount: true,
        },
        _count: {
          id: true,
        },
      });

      for (const earning of earnings) {
        if (!earning.driver_id) continue;

        await this.prisma.driverDailyEarning.create({
          data: {
            driver_id: earning.driver_id,
            date: yesterday,
            total_bookings: earning._count.id,
            total_earnings: earning._sum.driver_earning || 0,
            total_tips: earning._sum.tip_amount || 0,
            created_at: new Date(),
          },
        });
      }

      this.logger.log(`Calculated daily earnings for ${earnings.length} drivers`);
    } catch (error) {
      this.logger.error(`calculateDailyEarnings failed: ${error.message}`);
    }
  }

  /**
   * Update driver ratings
   * Runs every hour
   */
  @Cron(CronExpression.EVERY_HOUR)
  async updateDriverRatings() {
    this.logger.debug('Running: updateDriverRatings');

    try {
      // Get drivers who received ratings in the last hour
      const oneHourAgo = new Date(Date.now() - 60 * 60 * 1000);

      const recentRatings = await this.prisma.rating.findMany({
        where: {
          created_at: { gte: oneHourAgo },
          driver_id: { not: null },
        },
        select: { driver_id: true },
        distinct: ['driver_id'],
      });

      for (const rating of recentRatings) {
        if (!rating.driver_id) continue;

        // Calculate new average
        const avgRating = await this.prisma.rating.aggregate({
          where: { driver_id: rating.driver_id },
          _avg: { rating: true },
          _count: { id: true },
        });

        await this.prisma.driver.update({
          where: { id: rating.driver_id },
          data: {
            rating: avgRating._avg.rating || 0,
            total_ratings: avgRating._count.id,
          },
        });
      }

      this.logger.debug(`Updated ratings for ${recentRatings.length} drivers`);
    } catch (error) {
      this.logger.error(`updateDriverRatings failed: ${error.message}`);
    }
  }

  /**
   * Reset driver daily stats
   * Runs daily at midnight
   */
  @Cron('0 0 * * *')
  async resetDailyStats() {
    this.logger.debug('Running: resetDailyStats');

    try {
      await this.prisma.driver.updateMany({
        data: {
          daily_trips: 0,
          daily_earnings: 0,
        },
      });

      this.logger.log('Reset daily stats for all drivers');
    } catch (error) {
      this.logger.error(`resetDailyStats failed: ${error.message}`);
    }
  }

  /**
   * Check and update driver levels/tiers
   * Runs weekly on Sunday at midnight
   */
  @Cron('0 0 * * 0')
  async updateDriverLevels() {
    this.logger.debug('Running: updateDriverLevels');

    try {
      // Get driver stats for the past month
      const oneMonthAgo = new Date();
      oneMonthAgo.setMonth(oneMonthAgo.getMonth() - 1);

      const drivers = await this.prisma.driver.findMany({
        where: { is_active: true },
        include: {
          _count: {
            select: {
              bookings: {
                where: {
                  status: 'completed',
                  created_at: { gte: oneMonthAgo },
                },
              },
            },
          },
        },
      });

      for (const driver of drivers) {
        const completedTrips = driver._count.bookings;
        let newLevel = 'bronze';

        if (completedTrips >= 200 && driver.rating >= 4.8) {
          newLevel = 'platinum';
        } else if (completedTrips >= 100 && driver.rating >= 4.5) {
          newLevel = 'gold';
        } else if (completedTrips >= 50 && driver.rating >= 4.0) {
          newLevel = 'silver';
        }

        if (driver.level !== newLevel) {
          await this.prisma.driver.update({
            where: { id: driver.id },
            data: { level: newLevel },
          });
          this.logger.log(`Driver ${driver.id} level changed: ${driver.level} -> ${newLevel}`);
        }
      }
    } catch (error) {
      this.logger.error(`updateDriverLevels failed: ${error.message}`);
    }
  }
}
