import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class BookingTasksService implements OnModuleInit {
  private readonly logger = new Logger(BookingTasksService.name);

  constructor(private prisma: PrismaService) {}

  onModuleInit() {
    this.logger.log('Booking Tasks initialized');
  }

  /**
   * Cancel expired pending bookings
   * Runs every 5 minutes
   */
  @Cron(CronExpression.EVERY_5_MINUTES)
  async cancelExpiredBookings() {
    this.logger.debug('Running: cancelExpiredBookings');

    try {
      const expirationTime = new Date(Date.now() - 10 * 60 * 1000); // 10 minutes

      const result = await this.prisma.booking.updateMany({
        where: {
          status: 'pending',
          created_at: { lt: expirationTime },
        },
        data: {
          status: 'cancelled',
          cancelled_reason: 'auto_expired',
          cancelled_at: new Date(),
        },
      });

      if (result.count > 0) {
        this.logger.log(`Cancelled ${result.count} expired bookings`);
      }
    } catch (error) {
      this.logger.error(`cancelExpiredBookings failed: ${error.message}`);
    }
  }

  /**
   * Auto-complete bookings that have been in progress for too long
   * Runs every 15 minutes
   */
  @Cron(CronExpression.EVERY_30_MINUTES)
  async autoCompleteStaleBookings() {
    this.logger.debug('Running: autoCompleteStaleBookings');

    try {
      const staleTime = new Date(Date.now() - 4 * 60 * 60 * 1000); // 4 hours

      const staleBookings = await this.prisma.booking.findMany({
        where: {
          status: 'in_progress',
          ride_started_at: { lt: staleTime },
        },
        select: { id: true, booking_number: true },
      });

      for (const booking of staleBookings) {
        await this.prisma.booking.update({
          where: { id: booking.id },
          data: {
            status: 'completed',
            ride_ended_at: new Date(),
            // Mark as auto-completed
          },
        });
        this.logger.warn(`Auto-completed stale booking: ${booking.booking_number}`);
      }
    } catch (error) {
      this.logger.error(`autoCompleteStaleBookings failed: ${error.message}`);
    }
  }

  /**
   * Update booking statistics
   * Runs every hour
   */
  @Cron(CronExpression.EVERY_HOUR)
  async updateBookingStats() {
    this.logger.debug('Running: updateBookingStats');

    try {
      const today = new Date();
      today.setHours(0, 0, 0, 0);

      // Get all merchants
      const merchants = await this.prisma.merchant.findMany({
        select: { id: true },
      });

      for (const merchant of merchants) {
        const stats = await this.prisma.booking.groupBy({
          by: ['status'],
          where: {
            merchant_id: merchant.id,
            created_at: { gte: today },
          },
          _count: { id: true },
        });

        // Store in cache or stats table
        this.logger.debug(`Stats for merchant ${merchant.id}: ${JSON.stringify(stats)}`);
      }
    } catch (error) {
      this.logger.error(`updateBookingStats failed: ${error.message}`);
    }
  }

  /**
   * Check for scheduled bookings and send reminders
   * Runs every 10 minutes
   */
  @Cron(CronExpression.EVERY_10_MINUTES)
  async processScheduledBookings() {
    this.logger.debug('Running: processScheduledBookings');

    try {
      const now = new Date();
      const reminderTime = new Date(Date.now() + 30 * 60 * 1000); // 30 minutes ahead

      // Find scheduled bookings that need reminders
      const upcomingBookings = await this.prisma.booking.findMany({
        where: {
          is_scheduled: true,
          status: 'scheduled',
          scheduled_time: {
            gte: now,
            lte: reminderTime,
          },
          reminder_sent: false,
        },
        include: {
          user: { select: { id: true, phone: true, email: true } },
          driver: { select: { id: true, phone: true, name: true } },
        },
      });

      for (const booking of upcomingBookings) {
        // Mark reminder as sent
        await this.prisma.booking.update({
          where: { id: booking.id },
          data: { reminder_sent: true },
        });

        // TODO: Send notification to user and driver
        this.logger.log(`Reminder sent for scheduled booking: ${booking.booking_number}`);
      }

      // Activate scheduled bookings that are due
      const dueTime = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes ahead

      const dueBookings = await this.prisma.booking.updateMany({
        where: {
          is_scheduled: true,
          status: 'scheduled',
          scheduled_time: { lte: dueTime },
        },
        data: {
          status: 'pending',
        },
      });

      if (dueBookings.count > 0) {
        this.logger.log(`Activated ${dueBookings.count} scheduled bookings`);
      }
    } catch (error) {
      this.logger.error(`processScheduledBookings failed: ${error.message}`);
    }
  }

  /**
   * Calculate and update driver assignments
   * Runs every 2 minutes for pending bookings
   */
  @Cron('*/2 * * * *')
  async reassignPendingBookings() {
    this.logger.debug('Running: reassignPendingBookings');

    try {
      const unassignedTime = new Date(Date.now() - 3 * 60 * 1000); // 3 minutes

      // Find bookings that need reassignment
      const pendingBookings = await this.prisma.booking.findMany({
        where: {
          status: 'pending',
          driver_id: null,
          is_scheduled: false,
          created_at: { lt: unassignedTime },
          reassignment_count: { lt: 3 }, // Max 3 reassignments
        },
        select: {
          id: true,
          booking_number: true,
          merchant_id: true,
          pickup_lat: true,
          pickup_lng: true,
          reassignment_count: true,
        },
        take: 10,
      });

      // TODO: Implement driver matching logic
      this.logger.debug(`Found ${pendingBookings.length} bookings needing reassignment`);
    } catch (error) {
      this.logger.error(`reassignPendingBookings failed: ${error.message}`);
    }
  }
}
