import {
  Controller,
  Get,
  Post,
  Param,
  UseGuards,
} from '@nestjs/common';
import { CronService } from './cron.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { AdminGuard } from '../../common/guards/admin.guard';

@Controller('admin/cron')
@UseGuards(JwtAuthGuard, AdminGuard)
export class CronController {
  constructor(private readonly cronService: CronService) {}

  /**
   * Get all scheduled tasks
   */
  @Get('tasks')
  async getTasks() {
    return this.cronService.getTasks();
  }

  /**
   * Get task details
   */
  @Get('tasks/:name')
  async getTask(@Param('name') name: string) {
    const task = this.cronService.getTask(name);
    if (!task) {
      return { success: false, message: 'Task not found' };
    }
    return task;
  }

  /**
   * Run a task manually
   */
  @Post('tasks/:name/run')
  async runTask(@Param('name') name: string) {
    return this.cronService.runTask(name);
  }

  /**
   * Start a task
   */
  @Post('tasks/:name/start')
  async startTask(@Param('name') name: string) {
    const started = this.cronService.startTask(name);
    return { success: started };
  }

  /**
   * Stop a task
   */
  @Post('tasks/:name/stop')
  async stopTask(@Param('name') name: string) {
    const stopped = this.cronService.stopTask(name);
    return { success: stopped };
  }
}
