import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

interface CorporateAccount {
  id: number;
  name: string;
  email: string;
  phone: string;
  address: string;
  taxId: string;
  billingType: 'prepaid' | 'postpaid';
  creditLimit: number;
  currentBalance: number;
  discountPercentage: number;
  isActive: boolean;
  employeeCount: number;
}

interface CorporateEmployee {
  id: number;
  userId: number;
  corporateId: number;
  employeeId: string;
  department: string;
  monthlyLimit: number;
  currentMonthSpent: number;
  isActive: boolean;
}

@Injectable()
export class CorporateService {
  private readonly logger = new Logger(CorporateService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  // ============================================================================
  // CORPORATE ACCOUNT MANAGEMENT
  // ============================================================================

  /**
   * Create corporate account
   */
  async createCorporateAccount(
    merchantId: number,
    data: {
      name: string;
      email: string;
      phone: string;
      address: string;
      taxId?: string;
      billingType: 'prepaid' | 'postpaid';
      creditLimit?: number;
      discountPercentage?: number;
    },
  ): Promise<any> {
    // Check if email already exists
    const existing = await this.prisma.corporateAccount.findFirst({
      where: { email: data.email, merchant_id: merchantId },
    });

    if (existing) {
      throw new BadRequestException('Un compte avec cet email existe deja');
    }

    const account = await this.prisma.corporateAccount.create({
      data: {
        merchant_id: merchantId,
        name: data.name,
        email: data.email,
        phone: data.phone,
        address: data.address,
        tax_id: data.taxId,
        billing_type: data.billingType,
        credit_limit: data.creditLimit || 0,
        current_balance: 0,
        discount_percentage: data.discountPercentage || 0,
        is_active: 1,
        created_at: new Date(),
      },
    });

    this.logger.log(`Corporate account created: ${data.name} (#${account.id})`);
    return account;
  }

  /**
   * Get corporate account details
   */
  async getCorporateAccount(
    corporateId: number,
    merchantId: number,
  ): Promise<CorporateAccount | null> {
    const account = await this.prisma.corporateAccount.findFirst({
      where: { id: corporateId, merchant_id: merchantId },
      include: {
        _count: {
          select: { employees: { where: { is_active: 1 } } },
        },
      },
    });

    if (!account) return null;

    return {
      id: account.id,
      name: account.name,
      email: account.email,
      phone: account.phone,
      address: account.address,
      taxId: account.tax_id,
      billingType: account.billing_type as 'prepaid' | 'postpaid',
      creditLimit: Number(account.credit_limit),
      currentBalance: Number(account.current_balance),
      discountPercentage: Number(account.discount_percentage),
      isActive: account.is_active === 1,
      employeeCount: account._count.employees,
    };
  }

  /**
   * Update corporate account
   */
  async updateCorporateAccount(
    corporateId: number,
    merchantId: number,
    data: Partial<{
      name: string;
      email: string;
      phone: string;
      address: string;
      taxId: string;
      billingType: 'prepaid' | 'postpaid';
      creditLimit: number;
      discountPercentage: number;
      isActive: boolean;
    }>,
  ): Promise<any> {
    return this.prisma.corporateAccount.update({
      where: { id: corporateId },
      data: {
        ...(data.name && { name: data.name }),
        ...(data.email && { email: data.email }),
        ...(data.phone && { phone: data.phone }),
        ...(data.address && { address: data.address }),
        ...(data.taxId !== undefined && { tax_id: data.taxId }),
        ...(data.billingType && { billing_type: data.billingType }),
        ...(data.creditLimit !== undefined && { credit_limit: data.creditLimit }),
        ...(data.discountPercentage !== undefined && { discount_percentage: data.discountPercentage }),
        ...(data.isActive !== undefined && { is_active: data.isActive ? 1 : 0 }),
        updated_at: new Date(),
      },
    });
  }

  /**
   * Add funds to corporate account (prepaid)
   */
  async addFunds(
    corporateId: number,
    merchantId: number,
    amount: number,
    paymentReference: string,
  ): Promise<{ newBalance: number }> {
    const account = await this.prisma.corporateAccount.findFirst({
      where: { id: corporateId, merchant_id: merchantId },
    });

    if (!account) {
      throw new NotFoundException('Compte corporate non trouve');
    }

    // Update balance
    const updated = await this.prisma.corporateAccount.update({
      where: { id: corporateId },
      data: {
        current_balance: { increment: amount },
      },
    });

    // Record transaction
    await this.prisma.corporateTransaction.create({
      data: {
        corporate_id: corporateId,
        merchant_id: merchantId,
        type: 'credit',
        amount,
        description: 'Recharge de compte',
        payment_reference: paymentReference,
        balance_after: Number(updated.current_balance),
        created_at: new Date(),
      },
    });

    this.logger.log(`Corporate #${corporateId} credited ${amount}`);
    return { newBalance: Number(updated.current_balance) };
  }

  // ============================================================================
  // EMPLOYEE MANAGEMENT
  // ============================================================================

  /**
   * Add employee to corporate account
   */
  async addEmployee(
    corporateId: number,
    merchantId: number,
    data: {
      userId: number;
      employeeId?: string;
      department?: string;
      monthlyLimit?: number;
    },
  ): Promise<any> {
    // Verify corporate account
    const account = await this.prisma.corporateAccount.findFirst({
      where: { id: corporateId, merchant_id: merchantId },
    });

    if (!account) {
      throw new NotFoundException('Compte corporate non trouve');
    }

    // Check if user is already an employee
    const existing = await this.prisma.corporateEmployee.findFirst({
      where: {
        corporate_id: corporateId,
        user_id: data.userId,
      },
    });

    if (existing) {
      throw new BadRequestException('Cet utilisateur est deja employe');
    }

    const employee = await this.prisma.corporateEmployee.create({
      data: {
        corporate_id: corporateId,
        user_id: data.userId,
        employee_id: data.employeeId,
        department: data.department,
        monthly_limit: data.monthlyLimit || 0,
        current_month_spent: 0,
        is_active: 1,
        created_at: new Date(),
      },
    });

    // Update user's corporate status
    await this.prisma.user.update({
      where: { id: data.userId },
      data: {
        is_corporate: 1,
        corporate_id: corporateId,
      },
    });

    return employee;
  }

  /**
   * Remove employee from corporate account
   */
  async removeEmployee(
    corporateId: number,
    employeeId: number,
    merchantId: number,
  ): Promise<void> {
    const employee = await this.prisma.corporateEmployee.findFirst({
      where: { id: employeeId, corporate_id: corporateId },
    });

    if (!employee) {
      throw new NotFoundException('Employe non trouve');
    }

    // Deactivate employee
    await this.prisma.corporateEmployee.update({
      where: { id: employeeId },
      data: {
        is_active: 0,
        deactivated_at: new Date(),
      },
    });

    // Update user's corporate status
    await this.prisma.user.update({
      where: { id: employee.user_id },
      data: {
        is_corporate: 0,
        corporate_id: null,
      },
    });
  }

  /**
   * Get corporate employees
   */
  async getEmployees(
    corporateId: number,
    options?: { page?: number; limit?: number; activeOnly?: boolean },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { corporate_id: corporateId };
    if (options?.activeOnly) {
      where.is_active = 1;
    }

    const [data, total] = await Promise.all([
      this.prisma.corporateEmployee.findMany({
        where,
        skip,
        take: limit,
        include: {
          user: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              email: true,
              mobile: true,
            },
          },
        },
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.corporateEmployee.count({ where }),
    ]);

    return { data, total };
  }

  /**
   * Update employee monthly limit
   */
  async updateEmployeeLimit(
    corporateId: number,
    employeeId: number,
    monthlyLimit: number,
  ): Promise<any> {
    return this.prisma.corporateEmployee.update({
      where: { id: employeeId },
      data: { monthly_limit: monthlyLimit },
    });
  }

  // ============================================================================
  // BOOKING VALIDATION
  // ============================================================================

  /**
   * Check if employee can book (within limits)
   */
  async canEmployeeBook(
    userId: number,
    fareAmount: number,
  ): Promise<{ canBook: boolean; reason?: string; corporate?: any }> {
    const employee = await this.prisma.corporateEmployee.findFirst({
      where: { user_id: userId, is_active: 1 },
      include: { corporate: true },
    });

    if (!employee) {
      return { canBook: false, reason: 'Non employe corporate' };
    }

    const corporate = employee.corporate;

    // Check if corporate is active
    if (corporate.is_active !== 1) {
      return { canBook: false, reason: 'Compte corporate desactive' };
    }

    // Check employee monthly limit
    if (employee.monthly_limit > 0) {
      const remaining = employee.monthly_limit - Number(employee.current_month_spent);
      if (fareAmount > remaining) {
        return {
          canBook: false,
          reason: `Limite mensuelle depassee (${remaining} XOF restants)`,
        };
      }
    }

    // Check corporate balance/credit
    if (corporate.billing_type === 'prepaid') {
      if (fareAmount > Number(corporate.current_balance)) {
        return { canBook: false, reason: 'Solde corporate insuffisant' };
      }
    } else {
      // Postpaid - check credit limit
      const totalOwed = await this.getTotalOwed(corporate.id);
      if (totalOwed + fareAmount > Number(corporate.credit_limit)) {
        return { canBook: false, reason: 'Limite de credit atteinte' };
      }
    }

    return {
      canBook: true,
      corporate: {
        id: corporate.id,
        name: corporate.name,
        discountPercentage: Number(corporate.discount_percentage),
      },
    };
  }

  /**
   * Process corporate booking payment
   */
  async processBookingPayment(
    corporateId: number,
    userId: number,
    bookingId: number,
    amount: number,
    originalAmount: number,
    discountAmount: number,
  ): Promise<void> {
    const corporate = await this.prisma.corporateAccount.findUnique({
      where: { id: corporateId },
    });

    if (!corporate) {
      throw new NotFoundException('Compte corporate non trouve');
    }

    // Deduct from balance (prepaid) or add to outstanding (postpaid)
    if (corporate.billing_type === 'prepaid') {
      await this.prisma.corporateAccount.update({
        where: { id: corporateId },
        data: {
          current_balance: { decrement: amount },
        },
      });
    }

    // Record transaction
    await this.prisma.corporateTransaction.create({
      data: {
        corporate_id: corporateId,
        merchant_id: corporate.merchant_id,
        user_id: userId,
        booking_id: bookingId,
        type: 'debit',
        amount,
        original_amount: originalAmount,
        discount_amount: discountAmount,
        description: `Course #${bookingId}`,
        created_at: new Date(),
      },
    });

    // Update employee monthly spent
    await this.prisma.corporateEmployee.updateMany({
      where: { user_id: userId, corporate_id: corporateId },
      data: {
        current_month_spent: { increment: amount },
      },
    });

    this.logger.log(`Corporate booking payment: ${amount} for booking #${bookingId}`);
  }

  // ============================================================================
  // REPORTING
  // ============================================================================

  /**
   * Get corporate usage report
   */
  async getUsageReport(
    corporateId: number,
    startDate: Date,
    endDate: Date,
  ): Promise<{
    totalBookings: number;
    totalAmount: number;
    totalDiscount: number;
    byEmployee: any[];
    byDepartment: any[];
  }> {
    const transactions = await this.prisma.corporateTransaction.findMany({
      where: {
        corporate_id: corporateId,
        type: 'debit',
        created_at: {
          gte: startDate,
          lte: endDate,
        },
      },
      include: {
        user: {
          select: { first_name: true, last_name: true },
        },
      },
    });

    // Aggregate by employee
    const employeeMap = new Map<number, { name: string; bookings: number; amount: number }>();
    let totalAmount = 0;
    let totalDiscount = 0;

    for (const tx of transactions) {
      totalAmount += Number(tx.amount);
      totalDiscount += Number(tx.discount_amount) || 0;

      if (tx.user_id) {
        const existing = employeeMap.get(tx.user_id);
        if (existing) {
          existing.bookings++;
          existing.amount += Number(tx.amount);
        } else {
          employeeMap.set(tx.user_id, {
            name: `${tx.user?.first_name || ''} ${tx.user?.last_name || ''}`.trim(),
            bookings: 1,
            amount: Number(tx.amount),
          });
        }
      }
    }

    // Get department breakdown
    const employees = await this.prisma.corporateEmployee.findMany({
      where: { corporate_id: corporateId },
      select: { user_id: true, department: true },
    });

    const deptMap = new Map<string, { bookings: number; amount: number }>();
    for (const emp of employees) {
      const empStats = employeeMap.get(emp.user_id);
      if (empStats) {
        const dept = emp.department || 'Non specifie';
        const existing = deptMap.get(dept);
        if (existing) {
          existing.bookings += empStats.bookings;
          existing.amount += empStats.amount;
        } else {
          deptMap.set(dept, {
            bookings: empStats.bookings,
            amount: empStats.amount,
          });
        }
      }
    }

    return {
      totalBookings: transactions.length,
      totalAmount,
      totalDiscount,
      byEmployee: Array.from(employeeMap.entries()).map(([id, data]) => ({
        userId: id,
        ...data,
      })),
      byDepartment: Array.from(deptMap.entries()).map(([dept, data]) => ({
        department: dept,
        ...data,
      })),
    };
  }

  /**
   * Get total owed (for postpaid accounts)
   */
  private async getTotalOwed(corporateId: number): Promise<number> {
    const result = await this.prisma.corporateTransaction.aggregate({
      where: {
        corporate_id: corporateId,
        type: 'debit',
        is_paid: 0,
      },
      _sum: { amount: true },
    });

    return Number(result._sum?.amount) || 0;
  }

  /**
   * Generate invoice for postpaid account
   */
  async generateInvoice(
    corporateId: number,
    merchantId: number,
    startDate: Date,
    endDate: Date,
  ): Promise<any> {
    const corporate = await this.prisma.corporateAccount.findFirst({
      where: { id: corporateId, merchant_id: merchantId },
    });

    if (!corporate) {
      throw new NotFoundException('Compte corporate non trouve');
    }

    // Get unpaid transactions
    const transactions = await this.prisma.corporateTransaction.findMany({
      where: {
        corporate_id: corporateId,
        type: 'debit',
        is_paid: 0,
        created_at: {
          gte: startDate,
          lte: endDate,
        },
      },
    });

    const totalAmount = transactions.reduce((sum, tx) => sum + Number(tx.amount), 0);

    // Create invoice
    const invoice = await this.prisma.corporateInvoice.create({
      data: {
        corporate_id: corporateId,
        merchant_id: merchantId,
        invoice_number: await this.generateInvoiceNumber(merchantId),
        period_start: startDate,
        period_end: endDate,
        total_bookings: transactions.length,
        total_amount: totalAmount,
        status: 'pending',
        due_date: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days
        created_at: new Date(),
      },
    });

    // Mark transactions as invoiced
    await this.prisma.corporateTransaction.updateMany({
      where: {
        id: { in: transactions.map((t) => t.id) },
      },
      data: { invoice_id: invoice.id },
    });

    return invoice;
  }

  private async generateInvoiceNumber(merchantId: number): Promise<string> {
    const date = new Date();
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');

    const count = await this.prisma.corporateInvoice.count({
      where: {
        merchant_id: merchantId,
        created_at: {
          gte: new Date(year, date.getMonth(), 1),
        },
      },
    });

    return `INV-${year}${month}-${String(count + 1).padStart(4, '0')}`;
  }

  /**
   * Reset monthly employee limits (called by cron job)
   */
  async resetMonthlyLimits(): Promise<void> {
    await this.prisma.corporateEmployee.updateMany({
      where: { is_active: 1 },
      data: { current_month_spent: 0 },
    });

    this.logger.log('Monthly employee limits reset');
  }
}
