import {
  Controller,
  Get,
  Post,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
} from '@nestjs/common';
import { CacheService } from './cache.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';

@Controller('admin/cache')
@UseGuards(JwtAuthGuard)
export class CacheController {
  constructor(private readonly cacheService: CacheService) {}

  /**
   * Get cache statistics
   */
  @Get('stats')
  async getStats() {
    return this.cacheService.getStats();
  }

  /**
   * Get cache value
   */
  @Get('get/:key')
  async getValue(@Param('key') key: string) {
    const value = await this.cacheService.get(key);
    const ttl = await this.cacheService.ttl(key);

    return {
      key,
      exists: value !== null,
      value,
      ttl,
    };
  }

  /**
   * Set cache value
   */
  @Post('set')
  async setValue(
    @Body() body: { key: string; value: any; ttl?: number; tags?: string[] },
  ) {
    const success = await this.cacheService.set(body.key, body.value, {
      ttl: body.ttl,
      tags: body.tags,
    });

    return {
      success,
      key: body.key,
    };
  }

  /**
   * Delete cache value
   */
  @Delete('delete/:key')
  async deleteValue(@Param('key') key: string) {
    const success = await this.cacheService.delete(key);

    return {
      success,
      key,
    };
  }

  /**
   * Find keys matching pattern
   */
  @Get('keys')
  async findKeys(@Query('pattern') pattern: string = '*') {
    const keys = await this.cacheService.keys(pattern);

    return {
      pattern,
      count: keys.length,
      keys: keys.slice(0, 100), // Limit to 100 keys
    };
  }

  /**
   * Delete keys matching pattern
   */
  @Delete('pattern')
  async deletePattern(@Body() body: { pattern: string }) {
    const deleted = await this.cacheService.deletePattern(body.pattern);

    return {
      success: true,
      pattern: body.pattern,
      deleted,
    };
  }

  /**
   * Invalidate cache by tag
   */
  @Post('invalidate/tag')
  async invalidateTag(@Body() body: { tag: string }) {
    const invalidated = await this.cacheService.invalidateTag(body.tag);

    return {
      success: true,
      tag: body.tag,
      invalidated,
    };
  }

  /**
   * Invalidate multiple tags
   */
  @Post('invalidate/tags')
  async invalidateTags(@Body() body: { tags: string[] }) {
    const invalidated = await this.cacheService.invalidateTags(body.tags);

    return {
      success: true,
      tags: body.tags,
      invalidated,
    };
  }

  /**
   * Flush entire cache (dangerous!)
   */
  @Post('flush')
  async flushCache() {
    const success = await this.cacheService.flush();

    return {
      success,
      message: success ? 'Cache flushed successfully' : 'Failed to flush cache',
    };
  }

  /**
   * Reset statistics
   */
  @Post('stats/reset')
  resetStats() {
    this.cacheService.resetStats();

    return {
      success: true,
      message: 'Statistics reset',
    };
  }

  /**
   * Common cache operations
   */
  @Post('invalidate/entity')
  async invalidateEntity(
    @Body() body: { entity: string; entityId?: number | string },
  ) {
    const patterns = [];

    if (body.entityId) {
      patterns.push(`${body.entity}:${body.entityId}:*`);
      patterns.push(`*:${body.entity}:${body.entityId}*`);
    } else {
      patterns.push(`${body.entity}:*`);
      patterns.push(`*:${body.entity}:*`);
    }

    let totalDeleted = 0;
    for (const pattern of patterns) {
      totalDeleted += await this.cacheService.deletePattern(pattern);
    }

    return {
      success: true,
      entity: body.entity,
      entityId: body.entityId,
      deleted: totalDeleted,
    };
  }

  /**
   * Warmup common caches
   */
  @Post('warmup')
  async warmupCache(@Body() body: { types?: string[] }) {
    const types = body.types || ['settings', 'vehicle_types', 'pricing'];
    const warmedUp: string[] = [];

    // This would call the appropriate services to warm up caches
    // For now, just return the types that would be warmed up
    for (const type of types) {
      warmedUp.push(type);
    }

    return {
      success: true,
      warmedUp,
      message: `Cache warmup initiated for: ${warmedUp.join(', ')}`,
    };
  }
}
