import { Injectable, Logger, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// Business segment types
export enum SegmentType {
  TAXI = 'taxi',
  RENTAL = 'rental',
  OUTSTATION = 'outstation',
  AIRPORT = 'airport',
  SCHEDULED = 'scheduled',
  CORPORATE = 'corporate',
  DELIVERY = 'delivery',
  HANDYMAN = 'handyman',
}

interface ServiceConfig {
  id: number;
  name: string;
  type: SegmentType;
  description: string;
  icon: string;
  isActive: boolean;
  minFare: number;
  baseFare: number;
  perKmRate: number;
  perMinuteRate: number;
  cancellationFee: number;
  vehicleTypes: VehicleTypeConfig[];
}

interface VehicleTypeConfig {
  id: number;
  name: string;
  description: string;
  icon: string;
  capacity: number;
  baseFare: number;
  perKmRate: number;
  perMinuteRate: number;
  minFare: number;
  isActive: boolean;
}

interface RentalPackage {
  id: number;
  name: string;
  hours: number;
  kilometers: number;
  baseFare: number;
  extraHourRate: number;
  extraKmRate: number;
  vehicleTypeId: number;
}

interface OutstationConfig {
  id: number;
  name: string;
  oneWayMultiplier: number;
  roundTripMultiplier: number;
  driverAllowancePerDay: number;
  nightChargePerNight: number;
  minKilometers: number;
}

@Injectable()
export class BusinessSegmentService {
  private readonly logger = new Logger(BusinessSegmentService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Get all available services for a merchant
   */
  async getAvailableServices(merchantId: number): Promise<ServiceConfig[]> {
    const services = await this.prisma.serviceType.findMany({
      where: {
        merchant_id: merchantId,
        is_active: 1,
      },
      include: {
        vehicleTypes: {
          where: { is_active: 1 },
          orderBy: { sort_order: 'asc' },
        },
      },
      orderBy: { sort_order: 'asc' },
    });

    return services.map((service) => ({
      id: service.id,
      name: service.name,
      type: service.type as SegmentType,
      description: service.description,
      icon: service.icon,
      isActive: service.is_active === 1,
      minFare: Number(service.min_fare) || 0,
      baseFare: Number(service.base_fare) || 0,
      perKmRate: Number(service.per_km_rate) || 0,
      perMinuteRate: Number(service.per_minute_rate) || 0,
      cancellationFee: Number(service.cancellation_fee) || 0,
      vehicleTypes: service.vehicleTypes.map((vt) => ({
        id: vt.id,
        name: vt.name,
        description: vt.description,
        icon: vt.icon,
        capacity: vt.capacity,
        baseFare: Number(vt.base_fare) || 0,
        perKmRate: Number(vt.per_km_rate) || 0,
        perMinuteRate: Number(vt.per_minute_rate) || 0,
        minFare: Number(vt.min_fare) || 0,
        isActive: vt.is_active === 1,
      })),
    }));
  }

  /**
   * Get service by type
   */
  async getServiceByType(
    merchantId: number,
    type: SegmentType,
  ): Promise<ServiceConfig | null> {
    const service = await this.prisma.serviceType.findFirst({
      where: {
        merchant_id: merchantId,
        type,
        is_active: 1,
      },
      include: {
        vehicleTypes: {
          where: { is_active: 1 },
          orderBy: { sort_order: 'asc' },
        },
      },
    });

    if (!service) return null;

    return {
      id: service.id,
      name: service.name,
      type: service.type as SegmentType,
      description: service.description,
      icon: service.icon,
      isActive: service.is_active === 1,
      minFare: Number(service.min_fare) || 0,
      baseFare: Number(service.base_fare) || 0,
      perKmRate: Number(service.per_km_rate) || 0,
      perMinuteRate: Number(service.per_minute_rate) || 0,
      cancellationFee: Number(service.cancellation_fee) || 0,
      vehicleTypes: service.vehicleTypes.map((vt) => ({
        id: vt.id,
        name: vt.name,
        description: vt.description,
        icon: vt.icon,
        capacity: vt.capacity,
        baseFare: Number(vt.base_fare) || 0,
        perKmRate: Number(vt.per_km_rate) || 0,
        perMinuteRate: Number(vt.per_minute_rate) || 0,
        minFare: Number(vt.min_fare) || 0,
        isActive: vt.is_active === 1,
      })),
    };
  }

  /**
   * Get vehicle types for a service
   */
  async getVehicleTypes(
    merchantId: number,
    serviceTypeId?: number,
  ): Promise<VehicleTypeConfig[]> {
    const where: any = {
      merchant_id: merchantId,
      is_active: 1,
    };

    if (serviceTypeId) {
      where.service_type_id = serviceTypeId;
    }

    const vehicleTypes = await this.prisma.vehicleType.findMany({
      where,
      orderBy: { sort_order: 'asc' },
    });

    return vehicleTypes.map((vt) => ({
      id: vt.id,
      name: vt.name,
      description: vt.description,
      icon: vt.icon,
      capacity: vt.capacity,
      baseFare: Number(vt.base_fare) || 0,
      perKmRate: Number(vt.per_km_rate) || 0,
      perMinuteRate: Number(vt.per_minute_rate) || 0,
      minFare: Number(vt.min_fare) || 0,
      isActive: vt.is_active === 1,
    }));
  }

  // ============================================================================
  // RENTAL SERVICE
  // ============================================================================

  /**
   * Get rental packages
   */
  async getRentalPackages(
    merchantId: number,
    vehicleTypeId?: number,
  ): Promise<RentalPackage[]> {
    const where: any = {
      merchant_id: merchantId,
      is_active: 1,
    };

    if (vehicleTypeId) {
      where.vehicle_type_id = vehicleTypeId;
    }

    const packages = await this.prisma.rentalPackage.findMany({
      where,
      orderBy: [{ hours: 'asc' }, { kilometers: 'asc' }],
    });

    return packages.map((pkg) => ({
      id: pkg.id,
      name: pkg.name,
      hours: pkg.hours,
      kilometers: pkg.kilometers,
      baseFare: Number(pkg.base_fare),
      extraHourRate: Number(pkg.extra_hour_rate),
      extraKmRate: Number(pkg.extra_km_rate),
      vehicleTypeId: pkg.vehicle_type_id,
    }));
  }

  /**
   * Calculate rental fare
   */
  async calculateRentalFare(
    merchantId: number,
    packageId: number,
    actualHours: number,
    actualKilometers: number,
  ): Promise<{
    baseFare: number;
    extraHoursFare: number;
    extraKmFare: number;
    totalFare: number;
  }> {
    const pkg = await this.prisma.rentalPackage.findFirst({
      where: { id: packageId, merchant_id: merchantId },
    });

    if (!pkg) {
      throw new NotFoundException('Package non trouve');
    }

    const baseFare = Number(pkg.base_fare);

    // Calculate extra hours
    const extraHours = Math.max(0, actualHours - pkg.hours);
    const extraHoursFare = Math.round(extraHours * Number(pkg.extra_hour_rate));

    // Calculate extra kilometers
    const extraKm = Math.max(0, actualKilometers - pkg.kilometers);
    const extraKmFare = Math.round(extraKm * Number(pkg.extra_km_rate));

    const totalFare = baseFare + extraHoursFare + extraKmFare;

    return {
      baseFare,
      extraHoursFare,
      extraKmFare,
      totalFare,
    };
  }

  // ============================================================================
  // OUTSTATION SERVICE
  // ============================================================================

  /**
   * Get outstation configuration
   */
  async getOutstationConfig(merchantId: number): Promise<OutstationConfig | null> {
    const config = await this.prisma.outstationConfig.findFirst({
      where: { merchant_id: merchantId, is_active: 1 },
    });

    if (!config) return null;

    return {
      id: config.id,
      name: config.name,
      oneWayMultiplier: Number(config.one_way_multiplier) || 1,
      roundTripMultiplier: Number(config.round_trip_multiplier) || 0.8,
      driverAllowancePerDay: Number(config.driver_allowance_per_day) || 0,
      nightChargePerNight: Number(config.night_charge_per_night) || 0,
      minKilometers: config.min_kilometers || 0,
    };
  }

  /**
   * Calculate outstation fare
   */
  async calculateOutstationFare(
    merchantId: number,
    vehicleTypeId: number,
    distanceKm: number,
    tripType: 'one_way' | 'round_trip',
    days: number = 1,
    nights: number = 0,
  ): Promise<{
    baseFare: number;
    distanceFare: number;
    driverAllowance: number;
    nightCharge: number;
    multiplier: number;
    totalFare: number;
  }> {
    const config = await this.getOutstationConfig(merchantId);
    if (!config) {
      throw new NotFoundException('Configuration outstation non disponible');
    }

    const vehicleType = await this.prisma.vehicleType.findFirst({
      where: { id: vehicleTypeId, merchant_id: merchantId },
    });

    if (!vehicleType) {
      throw new NotFoundException('Type de vehicule non trouve');
    }

    const baseFare = Number(vehicleType.base_fare) || 0;
    const perKmRate = Number(vehicleType.per_km_rate) || 0;

    // Calculate distance fare
    const effectiveDistance = Math.max(distanceKm, config.minKilometers);
    let distanceFare = effectiveDistance * perKmRate;

    // Apply trip type multiplier
    const multiplier = tripType === 'one_way'
      ? config.oneWayMultiplier
      : config.roundTripMultiplier;

    distanceFare = Math.round(distanceFare * multiplier);

    // Driver allowance
    const driverAllowance = days * config.driverAllowancePerDay;

    // Night charge
    const nightCharge = nights * config.nightChargePerNight;

    const totalFare = baseFare + distanceFare + driverAllowance + nightCharge;

    return {
      baseFare,
      distanceFare,
      driverAllowance,
      nightCharge,
      multiplier,
      totalFare: Math.round(totalFare),
    };
  }

  // ============================================================================
  // AIRPORT SERVICE
  // ============================================================================

  /**
   * Get airport zones
   */
  async getAirportZones(merchantId: number): Promise<any[]> {
    const zones = await this.prisma.airportZone.findMany({
      where: { merchant_id: merchantId, is_active: 1 },
      orderBy: { name: 'asc' },
    });

    return zones.map((zone) => ({
      id: zone.id,
      airportCode: zone.airport_code,
      airportName: zone.airport_name,
      latitude: Number(zone.latitude),
      longitude: Number(zone.longitude),
      radiusKm: zone.radius_km,
      pickupSurcharge: Number(zone.pickup_surcharge) || 0,
      dropoffSurcharge: Number(zone.dropoff_surcharge) || 0,
      waitingTimeIncluded: zone.waiting_time_included || 0,
    }));
  }

  /**
   * Check if location is in airport zone
   */
  async isInAirportZone(
    merchantId: number,
    latitude: number,
    longitude: number,
  ): Promise<{ isAirport: boolean; zone?: any; surcharge: number }> {
    const zones = await this.getAirportZones(merchantId);

    for (const zone of zones) {
      const distance = this.calculateDistance(
        latitude,
        longitude,
        zone.latitude,
        zone.longitude,
      );

      if (distance <= zone.radiusKm) {
        return {
          isAirport: true,
          zone,
          surcharge: zone.pickupSurcharge,
        };
      }
    }

    return { isAirport: false, surcharge: 0 };
  }

  // ============================================================================
  // SCHEDULED RIDES
  // ============================================================================

  /**
   * Get scheduled ride settings
   */
  async getScheduledRideSettings(merchantId: number): Promise<{
    minAdvanceMinutes: number;
    maxAdvanceDays: number;
    cancellationWindowMinutes: number;
    reminderMinutesBefore: number;
  }> {
    const merchant = await this.prisma.merchant.findUnique({
      where: { id: merchantId },
      select: {
        scheduled_min_advance_minutes: true,
        scheduled_max_advance_days: true,
        scheduled_cancellation_window: true,
        scheduled_reminder_minutes: true,
      },
    });

    return {
      minAdvanceMinutes: merchant?.scheduled_min_advance_minutes || 30,
      maxAdvanceDays: merchant?.scheduled_max_advance_days || 7,
      cancellationWindowMinutes: merchant?.scheduled_cancellation_window || 60,
      reminderMinutesBefore: merchant?.scheduled_reminder_minutes || 15,
    };
  }

  /**
   * Validate scheduled time
   */
  async validateScheduledTime(
    merchantId: number,
    scheduledTime: Date,
  ): Promise<{ valid: boolean; message?: string }> {
    const settings = await this.getScheduledRideSettings(merchantId);
    const now = new Date();

    const minTime = new Date(now.getTime() + settings.minAdvanceMinutes * 60 * 1000);
    const maxTime = new Date(now.getTime() + settings.maxAdvanceDays * 24 * 60 * 60 * 1000);

    if (scheduledTime < minTime) {
      return {
        valid: false,
        message: `La reservation doit etre au moins ${settings.minAdvanceMinutes} minutes a l'avance`,
      };
    }

    if (scheduledTime > maxTime) {
      return {
        valid: false,
        message: `La reservation ne peut pas depasser ${settings.maxAdvanceDays} jours a l'avance`,
      };
    }

    return { valid: true };
  }

  // ============================================================================
  // ADMIN ENDPOINTS
  // ============================================================================

  /**
   * Create or update service type
   */
  async upsertServiceType(
    merchantId: number,
    data: {
      id?: number;
      name: string;
      type: SegmentType;
      description?: string;
      icon?: string;
      baseFare?: number;
      perKmRate?: number;
      perMinuteRate?: number;
      minFare?: number;
      cancellationFee?: number;
      isActive?: boolean;
    },
  ): Promise<any> {
    if (data.id) {
      return this.prisma.serviceType.update({
        where: { id: data.id },
        data: {
          name: data.name,
          type: data.type,
          description: data.description,
          icon: data.icon,
          base_fare: data.baseFare,
          per_km_rate: data.perKmRate,
          per_minute_rate: data.perMinuteRate,
          min_fare: data.minFare,
          cancellation_fee: data.cancellationFee,
          is_active: data.isActive ? 1 : 0,
          updated_at: new Date(),
        },
      });
    }

    return this.prisma.serviceType.create({
      data: {
        merchant_id: merchantId,
        name: data.name,
        type: data.type,
        description: data.description,
        icon: data.icon,
        base_fare: data.baseFare,
        per_km_rate: data.perKmRate,
        per_minute_rate: data.perMinuteRate,
        min_fare: data.minFare,
        cancellation_fee: data.cancellationFee,
        is_active: data.isActive !== false ? 1 : 0,
        created_at: new Date(),
      },
    });
  }

  /**
   * Create or update vehicle type
   */
  async upsertVehicleType(
    merchantId: number,
    data: {
      id?: number;
      serviceTypeId?: number;
      name: string;
      description?: string;
      icon?: string;
      capacity?: number;
      baseFare?: number;
      perKmRate?: number;
      perMinuteRate?: number;
      minFare?: number;
      isActive?: boolean;
    },
  ): Promise<any> {
    if (data.id) {
      return this.prisma.vehicleType.update({
        where: { id: data.id },
        data: {
          service_type_id: data.serviceTypeId,
          name: data.name,
          description: data.description,
          icon: data.icon,
          capacity: data.capacity,
          base_fare: data.baseFare,
          per_km_rate: data.perKmRate,
          per_minute_rate: data.perMinuteRate,
          min_fare: data.minFare,
          is_active: data.isActive ? 1 : 0,
          updated_at: new Date(),
        },
      });
    }

    return this.prisma.vehicleType.create({
      data: {
        merchant_id: merchantId,
        service_type_id: data.serviceTypeId,
        name: data.name,
        description: data.description,
        icon: data.icon,
        capacity: data.capacity || 4,
        base_fare: data.baseFare,
        per_km_rate: data.perKmRate,
        per_minute_rate: data.perMinuteRate,
        min_fare: data.minFare,
        is_active: data.isActive !== false ? 1 : 0,
        created_at: new Date(),
      },
    });
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  private calculateDistance(
    lat1: number,
    lng1: number,
    lat2: number,
    lng2: number,
  ): number {
    const R = 6371;
    const dLat = this.toRad(lat2 - lat1);
    const dLng = this.toRad(lng2 - lng1);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) *
        Math.cos(this.toRad(lat2)) *
        Math.sin(dLng / 2) *
        Math.sin(dLng / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }

  private toRad(deg: number): number {
    return deg * (Math.PI / 180);
  }
}
