import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { BusinessSegmentService, SegmentType } from './business-segment.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';
import { Public } from '../../common/decorators/public.decorator';

@ApiTags('Business Segment')
@Controller('api/services')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class BusinessSegmentController {
  constructor(private businessSegmentService: BusinessSegmentService) {}

  // ============================================================================
  // PUBLIC ENDPOINTS
  // ============================================================================

  @Get()
  @Public()
  @ApiOperation({ summary: 'Get all available services' })
  async getServices(@MerchantId() merchantId: number) {
    return this.businessSegmentService.getAvailableServices(merchantId);
  }

  @Get('type/:type')
  @Public()
  @ApiOperation({ summary: 'Get service by type' })
  async getServiceByType(
    @Param('type') type: SegmentType,
    @MerchantId() merchantId: number,
  ) {
    return this.businessSegmentService.getServiceByType(merchantId, type);
  }

  @Get('vehicle-types')
  @Public()
  @ApiOperation({ summary: 'Get vehicle types' })
  async getVehicleTypes(
    @MerchantId() merchantId: number,
    @Query('service_type_id') serviceTypeId?: string,
  ) {
    return this.businessSegmentService.getVehicleTypes(
      merchantId,
      serviceTypeId ? parseInt(serviceTypeId) : undefined,
    );
  }

  // ============================================================================
  // RENTAL ENDPOINTS
  // ============================================================================

  @Get('rental/packages')
  @Public()
  @ApiOperation({ summary: 'Get rental packages' })
  async getRentalPackages(
    @MerchantId() merchantId: number,
    @Query('vehicle_type_id') vehicleTypeId?: string,
  ) {
    return this.businessSegmentService.getRentalPackages(
      merchantId,
      vehicleTypeId ? parseInt(vehicleTypeId) : undefined,
    );
  }

  @Post('rental/calculate')
  @ApiOperation({ summary: 'Calculate rental fare' })
  async calculateRentalFare(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      packageId: number;
      actualHours: number;
      actualKilometers: number;
    },
  ) {
    return this.businessSegmentService.calculateRentalFare(
      merchantId,
      body.packageId,
      body.actualHours,
      body.actualKilometers,
    );
  }

  // ============================================================================
  // OUTSTATION ENDPOINTS
  // ============================================================================

  @Get('outstation/config')
  @Public()
  @ApiOperation({ summary: 'Get outstation configuration' })
  async getOutstationConfig(@MerchantId() merchantId: number) {
    return this.businessSegmentService.getOutstationConfig(merchantId);
  }

  @Post('outstation/calculate')
  @ApiOperation({ summary: 'Calculate outstation fare' })
  async calculateOutstationFare(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      vehicleTypeId: number;
      distanceKm: number;
      tripType: 'one_way' | 'round_trip';
      days?: number;
      nights?: number;
    },
  ) {
    return this.businessSegmentService.calculateOutstationFare(
      merchantId,
      body.vehicleTypeId,
      body.distanceKm,
      body.tripType,
      body.days,
      body.nights,
    );
  }

  // ============================================================================
  // AIRPORT ENDPOINTS
  // ============================================================================

  @Get('airport/zones')
  @Public()
  @ApiOperation({ summary: 'Get airport zones' })
  async getAirportZones(@MerchantId() merchantId: number) {
    return this.businessSegmentService.getAirportZones(merchantId);
  }

  @Post('airport/check')
  @ApiOperation({ summary: 'Check if location is in airport zone' })
  async checkAirportZone(
    @MerchantId() merchantId: number,
    @Body() body: { latitude: number; longitude: number },
  ) {
    return this.businessSegmentService.isInAirportZone(
      merchantId,
      body.latitude,
      body.longitude,
    );
  }

  // ============================================================================
  // SCHEDULED RIDES ENDPOINTS
  // ============================================================================

  @Get('scheduled/settings')
  @ApiOperation({ summary: 'Get scheduled ride settings' })
  async getScheduledSettings(@MerchantId() merchantId: number) {
    return this.businessSegmentService.getScheduledRideSettings(merchantId);
  }

  @Post('scheduled/validate')
  @ApiOperation({ summary: 'Validate scheduled time' })
  async validateScheduledTime(
    @MerchantId() merchantId: number,
    @Body() body: { scheduledTime: string },
  ) {
    return this.businessSegmentService.validateScheduledTime(
      merchantId,
      new Date(body.scheduledTime),
    );
  }

  // ============================================================================
  // ADMIN ENDPOINTS
  // ============================================================================

  @Post('admin/service-type')
  @ApiOperation({ summary: 'Create or update service type (admin)' })
  async upsertServiceType(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      id?: number;
      name: string;
      type: SegmentType;
      description?: string;
      icon?: string;
      baseFare?: number;
      perKmRate?: number;
      perMinuteRate?: number;
      minFare?: number;
      cancellationFee?: number;
      isActive?: boolean;
    },
  ) {
    return this.businessSegmentService.upsertServiceType(merchantId, body);
  }

  @Post('admin/vehicle-type')
  @ApiOperation({ summary: 'Create or update vehicle type (admin)' })
  async upsertVehicleType(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      id?: number;
      serviceTypeId?: number;
      name: string;
      description?: string;
      icon?: string;
      capacity?: number;
      baseFare?: number;
      perKmRate?: number;
      perMinuteRate?: number;
      minFare?: number;
      isActive?: boolean;
    },
  ) {
    return this.businessSegmentService.upsertVehicleType(merchantId, body);
  }
}
