import {
  Injectable,
  NotFoundException,
  BadRequestException,
  ForbiddenException,
  Logger,
} from '@nestjs/common';
import { PrismaService } from '../../shared/database/prisma.service';
import { CreateBookingDto } from './dto/create-booking.dto';
import { EstimateBookingDto } from './dto/estimate-booking.dto';
import { RateBookingDto } from './dto/rate-booking.dto';
import { CancelBookingDto } from './dto/cancel-booking.dto';
import {
  BOOKING_STATUS,
  BOOKING_TYPE,
  PAYMENT_METHOD,
  PAYMENT_STATUS,
  DRIVER_SEARCH_CONFIG,
  isValidStatusTransition,
  canCancelBooking,
  shouldApplyCancellationFee,
  getStatusLabel,
  CANCELLABLE_STATUSES,
  ACTIVE_BOOKING_STATUSES,
} from './booking.constants';

interface NearbyDriver {
  id: number;
  distance: number;
  first_name: string;
  last_name: string;
  latitude: number;
  longitude: number;
  vehicle_type_id: number;
}

interface DriverSearchResult {
  found: boolean;
  driver?: NearbyDriver;
  message: string;
}

@Injectable()
export class BookingService {
  private readonly logger = new Logger(BookingService.name);

  constructor(private prisma: PrismaService) {}

  // =============================================================================
  // FARE ESTIMATION
  // =============================================================================

  async estimate(dto: EstimateBookingDto, merchantId: number) {
    // Get vehicle type pricing
    const vehicleType = await this.prisma.vehicleType.findFirst({
      where: {
        id: dto.vehicle_type_id,
        merchant_id: merchantId,
        vehicleTypeStatus: 1,
      },
    });

    if (!vehicleType) {
      throw new BadRequestException('Vehicle type not available');
    }

    // Get pricing from price_card_details (Laravel uses this)
    const priceCard = await this.prisma.priceCardDetail.findFirst({
      where: {
        merchant_id: merchantId,
        vehicle_type_id: dto.vehicle_type_id,
        status: 1,
      },
    });

    // Calculate distance using Haversine formula
    const distance = this.calculateDistance(
      dto.pickup_latitude,
      dto.pickup_longitude,
      dto.drop_latitude,
      dto.drop_longitude,
    );

    // Estimate time (assuming 30km/h average speed in urban areas)
    const estimatedTime = Math.ceil((distance / 30) * 60);

    // Calculate fare based on price card or defaults
    let baseFare = 500;
    let perKm = 100;
    let perMinute = 10;
    let minFare = 1000;

    if (priceCard) {
      baseFare = parseFloat(priceCard.base_fare?.toString() || '500');
      perKm = parseFloat(priceCard.distance_price?.toString() || '100');
      perMinute = parseFloat(priceCard.duration_price?.toString() || '10');
      minFare = parseFloat(priceCard.min_fare?.toString() || '1000');
    }

    // Calculate fare components
    const distanceFare = distance * perKm;
    const timeFare = estimatedTime * perMinute;
    let totalFare = baseFare + distanceFare + timeFare;

    // Check for surge pricing
    const surgeMultiplier = await this.getSurgeMultiplier(
      merchantId,
      dto.pickup_latitude,
      dto.pickup_longitude,
    );

    const surgeAmount = surgeMultiplier > 1 ? totalFare * (surgeMultiplier - 1) : 0;
    totalFare = totalFare * surgeMultiplier;

    // Apply minimum fare
    if (totalFare < minFare) {
      totalFare = minFare;
    }

    // Get merchant tax rate
    const merchant = await this.prisma.merchant.findUnique({
      where: { id: merchantId },
      select: { tax: true },
    });
    const taxRate = parseFloat(merchant?.tax || '0') / 100;
    const taxAmount = totalFare * taxRate;

    return {
      message: 'Fare estimate',
      data: {
        distance: Math.round(distance * 100) / 100,
        estimated_time: estimatedTime,
        base_fare: Math.round(baseFare),
        distance_fare: Math.round(distanceFare),
        time_fare: Math.round(timeFare),
        surge_multiplier: surgeMultiplier,
        surge_amount: Math.round(surgeAmount),
        sub_total: Math.round(totalFare),
        tax_amount: Math.round(taxAmount),
        total_fare: Math.round(totalFare + taxAmount),
        min_fare: minFare,
        currency: 'XOF',
        vehicle_type_id: dto.vehicle_type_id,
      },
    };
  }

  // =============================================================================
  // BOOKING CREATION
  // =============================================================================

  async create(userId: number, dto: CreateBookingDto, merchantId: number) {
    // Check if user has active booking
    const activeBooking = await this.prisma.booking.findFirst({
      where: {
        user_id: userId,
        booking_status: { in: ACTIVE_BOOKING_STATUSES },
      },
    });

    if (activeBooking) {
      throw new BadRequestException('You already have an active booking');
    }

    // Generate merchant booking ID (Laravel format)
    const merchantBookingId = `BK${merchantId}${Date.now()}`;

    // Calculate estimate
    const estimate = await this.estimate({
      ...dto,
      vehicle_type_id: dto.vehicle_type_id,
    }, merchantId);

    // Determine booking type
    const bookingType = dto.schedule_time
      ? BOOKING_TYPE.LATER
      : BOOKING_TYPE.NORMAL;

    // Initial status based on booking type
    const initialStatus = dto.schedule_time
      ? BOOKING_STATUS.SCHEDULED
      : BOOKING_STATUS.PENDING;

    // Create booking
    const booking = await this.prisma.booking.create({
      data: {
        merchant_id: merchantId,
        user_id: userId,
        merchant_booking_id: merchantBookingId,
        booking_status: initialStatus,
        booking_type: bookingType,
        vehicle_type_id: dto.vehicle_type_id,
        pickup_latitude: dto.pickup_latitude,
        pickup_longitude: dto.pickup_longitude,
        pickup_address: dto.pickup_address,
        drop_latitude: dto.drop_latitude,
        drop_longitude: dto.drop_longitude,
        drop_address: dto.drop_address,
        estimate_amount: estimate.data.total_fare,
        travel_distance: estimate.data.distance,
        travel_time: estimate.data.estimated_time,
        payment_method_id: dto.payment_method_id,
        booking_time: new Date(),
        schedule_time: dto.schedule_time ? new Date(dto.schedule_time) : null,
        promo_code_id: dto.promo_code_id || null,
        notes: dto.notes || null,
        payment_status: PAYMENT_STATUS.PENDING,
      },
      include: {
        user: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            UserPhone: true,
            profile_image: true,
          },
        },
      },
    });

    // Create booking transaction record
    await this.prisma.bookingTransaction.create({
      data: {
        booking_id: booking.id,
        merchant_id: merchantId,
        sub_total_before_discount: estimate.data.sub_total.toString(),
        surge_amount: estimate.data.surge_amount.toString(),
        tax_amount: estimate.data.tax_amount.toString(),
      },
    });

    // If immediate booking, start driver search
    if (!dto.schedule_time) {
      // Update status to searching
      await this.updateBookingStatus(booking.id, BOOKING_STATUS.SEARCHING);

      // Start async driver search (in production, this would be a queue job)
      this.startDriverSearch(booking.id, merchantId, dto.vehicle_type_id, {
        latitude: dto.pickup_latitude,
        longitude: dto.pickup_longitude,
      }).catch(err => this.logger.error(`Driver search failed: ${err.message}`));
    }

    return {
      message: 'Booking created successfully',
      data: {
        booking: {
          ...booking,
          status_label: getStatusLabel(booking.booking_status),
        },
        estimate: estimate.data,
      },
    };
  }

  // =============================================================================
  // DRIVER SEARCH ALGORITHM
  // =============================================================================

  private async startDriverSearch(
    bookingId: number,
    merchantId: number,
    vehicleTypeId: number,
    pickupLocation: { latitude: number; longitude: number },
  ): Promise<void> {
    const config = DRIVER_SEARCH_CONFIG;
    let currentRadius = config.INITIAL_RADIUS_KM;
    let searchRound = 0;
    const requestedDrivers = new Set<number>();

    this.logger.log(`Starting driver search for booking ${bookingId}`);

    while (searchRound < config.MAX_SEARCH_ROUNDS) {
      searchRound++;
      this.logger.log(`Search round ${searchRound}, radius: ${currentRadius}km`);

      // Check if booking is still in searching state
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
        select: { booking_status: true },
      });

      if (booking?.booking_status !== BOOKING_STATUS.SEARCHING) {
        this.logger.log(`Booking ${bookingId} is no longer searching`);
        return;
      }

      // Find nearby available drivers
      const nearbyDrivers = await this.findNearbyDrivers(
        merchantId,
        vehicleTypeId,
        pickupLocation,
        currentRadius,
        Array.from(requestedDrivers),
      );

      if (nearbyDrivers.length === 0) {
        this.logger.log(`No drivers found in radius ${currentRadius}km`);
        currentRadius = Math.min(currentRadius + config.RADIUS_INCREMENT_KM, config.MAX_RADIUS_KM);
        continue;
      }

      // Request drivers one by one
      for (const driver of nearbyDrivers.slice(0, config.MAX_DRIVERS_PER_ROUND)) {
        if (requestedDrivers.has(driver.id)) continue;
        requestedDrivers.add(driver.id);

        this.logger.log(`Requesting driver ${driver.id} for booking ${bookingId}`);

        // Create driver request record
        await this.prisma.driverRequest.create({
          data: {
            booking_id: bookingId,
            driver_id: driver.id,
            merchant_id: merchantId,
            status: 'pending',
            requested_at: new Date(),
            expires_at: new Date(Date.now() + config.REQUEST_TIMEOUT_SEC * 1000),
          },
        });

        // TODO: Send push notification to driver
        // await this.notificationService.sendToDriver(driver.id, {
        //   type: 'new_booking_request',
        //   booking_id: bookingId,
        //   pickup_address: booking.pickup_address,
        //   ...
        // });

        // Wait for driver response (timeout)
        const accepted = await this.waitForDriverResponse(
          bookingId,
          driver.id,
          config.REQUEST_TIMEOUT_SEC * 1000,
        );

        if (accepted) {
          this.logger.log(`Driver ${driver.id} accepted booking ${bookingId}`);
          return;
        }

        // Driver didn't accept, mark request as expired
        await this.prisma.driverRequest.updateMany({
          where: {
            booking_id: bookingId,
            driver_id: driver.id,
            status: 'pending',
          },
          data: { status: 'expired' },
        });
      }

      // Increase radius for next round
      currentRadius = Math.min(currentRadius + config.RADIUS_INCREMENT_KM, config.MAX_RADIUS_KM);
    }

    // No driver found after all rounds
    this.logger.log(`No driver found for booking ${bookingId} after ${searchRound} rounds`);
    await this.updateBookingStatus(bookingId, BOOKING_STATUS.NO_DRIVER_FOUND);

    // TODO: Notify user that no driver was found
  }

  private async findNearbyDrivers(
    merchantId: number,
    vehicleTypeId: number,
    location: { latitude: number; longitude: number },
    radiusKm: number,
    excludeDriverIds: number[],
  ): Promise<NearbyDriver[]> {
    // Get all online, free drivers with matching vehicle type
    const drivers = await this.prisma.driver.findMany({
      where: {
        merchant_id: merchantId,
        is_online: 1,
        free_busy: 2, // 2 = free, 1 = busy
        driverStatus: 1, // Active
        latitude: { not: null },
        longitude: { not: null },
        id: { notIn: excludeDriverIds },
        // Filter by vehicle type
        vehicles: {
          some: {
            vehicle_type_id: vehicleTypeId,
            is_active: 1,
          },
        },
      },
      select: {
        id: true,
        first_name: true,
        last_name: true,
        latitude: true,
        longitude: true,
        vehicles: {
          where: { vehicle_type_id: vehicleTypeId },
          select: { vehicle_type_id: true },
        },
      },
    });

    // Calculate distance and filter by radius
    const nearbyDrivers: NearbyDriver[] = [];

    for (const driver of drivers) {
      if (!driver.latitude || !driver.longitude) continue;

      const distance = this.calculateDistance(
        location.latitude,
        location.longitude,
        driver.latitude,
        driver.longitude,
      );

      if (distance <= radiusKm) {
        nearbyDrivers.push({
          id: driver.id,
          distance,
          first_name: driver.first_name || '',
          last_name: driver.last_name || '',
          latitude: driver.latitude,
          longitude: driver.longitude,
          vehicle_type_id: vehicleTypeId,
        });
      }
    }

    // Sort by distance (closest first)
    return nearbyDrivers.sort((a, b) => a.distance - b.distance);
  }

  private async waitForDriverResponse(
    bookingId: number,
    driverId: number,
    timeoutMs: number,
  ): Promise<boolean> {
    const startTime = Date.now();

    while (Date.now() - startTime < timeoutMs) {
      // Check if driver accepted
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
        select: {
          booking_status: true,
          driver_id: true,
        },
      });

      // Booking was accepted by this driver
      if (booking?.driver_id === driverId &&
          booking.booking_status === BOOKING_STATUS.DRIVER_ACCEPTED) {
        return true;
      }

      // Booking was cancelled or assigned to another driver
      if (booking?.booking_status !== BOOKING_STATUS.SEARCHING) {
        return false;
      }

      // Wait 2 seconds before checking again
      await new Promise(resolve => setTimeout(resolve, 2000));
    }

    return false;
  }

  // =============================================================================
  // STATUS MANAGEMENT
  // =============================================================================

  private async updateBookingStatus(
    bookingId: number,
    newStatus: number,
  ): Promise<void> {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
      select: { booking_status: true },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    // Validate transition
    if (!isValidStatusTransition(booking.booking_status, newStatus)) {
      throw new BadRequestException(
        `Invalid status transition from ${getStatusLabel(booking.booking_status)} to ${getStatusLabel(newStatus)}`,
      );
    }

    await this.prisma.booking.update({
      where: { id: bookingId },
      data: { booking_status: newStatus },
    });
  }

  // =============================================================================
  // BOOKING QUERIES
  // =============================================================================

  async getBooking(bookingId: number, userId: number, userType: string) {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
      include: {
        user: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            UserPhone: true,
            profile_image: true,
          },
        },
        driver: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            phoneNumber: true,
            profile_image: true,
            latitude: true,
            longitude: true,
          },
        },
        ratings: true,
        transaction: true,
      },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    // Check access
    if (userType === 'user' && booking.user_id !== userId) {
      throw new ForbiddenException('Access denied');
    }
    if (userType === 'driver' && booking.driver_id !== userId) {
      throw new ForbiddenException('Access denied');
    }

    return {
      message: 'Booking details',
      data: {
        ...booking,
        status_label: getStatusLabel(booking.booking_status),
        can_cancel: canCancelBooking(booking.booking_status),
      },
    };
  }

  async getUserBookings(
    userId: number,
    merchantId: number,
    status?: string,
    page = 1,
    limit = 10,
  ) {
    const skip = (page - 1) * limit;

    const where: any = {
      user_id: userId,
      merchant_id: merchantId,
    };

    // Filter by status category
    if (status === 'active') {
      where.booking_status = { in: ACTIVE_BOOKING_STATUSES };
    } else if (status === 'completed') {
      where.booking_status = BOOKING_STATUS.RIDE_COMPLETED;
    } else if (status === 'cancelled') {
      where.booking_status = {
        in: [
          BOOKING_STATUS.CANCELLED_BY_USER,
          BOOKING_STATUS.CANCELLED_BY_DRIVER,
          BOOKING_STATUS.CANCELLED_BY_SYSTEM,
        ],
      };
    }

    const [bookings, total] = await Promise.all([
      this.prisma.booking.findMany({
        where,
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
        include: {
          driver: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              profile_image: true,
            },
          },
        },
      }),
      this.prisma.booking.count({ where }),
    ]);

    return {
      message: 'User bookings',
      data: {
        bookings: bookings.map(b => ({
          ...b,
          status_label: getStatusLabel(b.booking_status),
        })),
        pagination: {
          page,
          limit,
          total,
          total_pages: Math.ceil(total / limit),
        },
      },
    };
  }

  // =============================================================================
  // USER ACTIONS
  // =============================================================================

  async cancel(
    bookingId: number,
    userId: number,
    userType: string,
    dto: CancelBookingDto,
  ) {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
      include: { transaction: true },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    // Check if booking can be cancelled
    if (!canCancelBooking(booking.booking_status)) {
      throw new BadRequestException(
        `Cannot cancel booking in status: ${getStatusLabel(booking.booking_status)}`,
      );
    }

    // Verify ownership
    if (userType === 'user' && booking.user_id !== userId) {
      throw new ForbiddenException('Access denied');
    }
    if (userType === 'driver' && booking.driver_id !== userId) {
      throw new ForbiddenException('Access denied');
    }

    // Determine new status based on who cancelled
    const newStatus = userType === 'user'
      ? BOOKING_STATUS.CANCELLED_BY_USER
      : BOOKING_STATUS.CANCELLED_BY_DRIVER;

    // Calculate cancellation fee if applicable
    let cancellationFee = 0;
    if (shouldApplyCancellationFee(booking.booking_status)) {
      // Get merchant cancellation settings
      const merchant = await this.prisma.merchant.findUnique({
        where: { id: booking.merchant_id },
        select: { cancel_charges: true },
      });

      if (merchant?.cancel_charges) {
        cancellationFee = merchant.cancel_charges;
      }
    }

    // Update booking status
    await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        booking_status: newStatus,
        cancelled_time: new Date(),
        cancel_reason: dto.reason,
      },
    });

    // Update transaction with cancellation charge
    if (cancellationFee > 0 && booking.transaction) {
      await this.prisma.bookingTransaction.update({
        where: { id: booking.transaction.id },
        data: {
          cancellation_charge_applied: cancellationFee.toString(),
        },
      });

      // TODO: Deduct from user wallet or charge card
    }

    // If driver was assigned, free the driver
    if (booking.driver_id) {
      await this.prisma.driver.update({
        where: { id: booking.driver_id },
        data: { free_busy: 2 }, // 2 = free
      });
    }

    // TODO: Send notifications to affected parties

    return {
      message: 'Booking cancelled successfully',
      data: {
        cancellation_fee: cancellationFee,
        refund_amount: cancellationFee > 0 ? 0 : (booking.estimate_amount || 0),
      },
    };
  }

  async rate(
    bookingId: number,
    userId: number,
    userType: string,
    dto: RateBookingDto,
  ) {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    if (booking.booking_status !== BOOKING_STATUS.RIDE_COMPLETED &&
        booking.booking_status !== BOOKING_STATUS.PAYMENT_COMPLETED) {
      throw new BadRequestException('Can only rate completed bookings');
    }

    // Create or update rating
    const existingRating = await this.prisma.bookingRating.findFirst({
      where: { booking_id: bookingId },
    });

    const ratingData: any = {
      booking_id: bookingId,
    };

    if (userType === 'user') {
      ratingData.driver_rating = dto.rating;
      ratingData.driver_comment = dto.comment;
      ratingData.driver_id = booking.driver_id;
    } else {
      ratingData.user_rating = dto.rating;
      ratingData.user_comment = dto.comment;
      ratingData.user_id = booking.user_id;
    }

    if (existingRating) {
      await this.prisma.bookingRating.update({
        where: { id: existingRating.id },
        data: ratingData,
      });
    } else {
      await this.prisma.bookingRating.create({
        data: ratingData,
      });
    }

    // Update driver/user average rating
    if (userType === 'user' && booking.driver_id) {
      await this.updateDriverRating(booking.driver_id);
    }

    return {
      message: 'Rating submitted successfully',
    };
  }

  private async updateDriverRating(driverId: number): Promise<void> {
    const result = await this.prisma.bookingRating.aggregate({
      where: {
        driver_id: driverId,
        driver_rating: { not: null },
      },
      _avg: { driver_rating: true },
      _count: true,
    });

    if (result._count > 0) {
      await this.prisma.driver.update({
        where: { id: driverId },
        data: {
          total_rating: result._count,
          average_rating: result._avg.driver_rating?.toString() || '0',
        },
      });
    }
  }

  // =============================================================================
  // DRIVER ACTIONS
  // =============================================================================

  async accept(bookingId: number, driverId: number) {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    if (booking.booking_status !== BOOKING_STATUS.SEARCHING) {
      throw new BadRequestException('Booking is not available for acceptance');
    }

    // Check if driver has pending request for this booking
    const driverRequest = await this.prisma.driverRequest.findFirst({
      where: {
        booking_id: bookingId,
        driver_id: driverId,
        status: 'pending',
      },
    });

    if (!driverRequest) {
      throw new BadRequestException('No pending request for this booking');
    }

    // Update driver request
    await this.prisma.driverRequest.update({
      where: { id: driverRequest.id },
      data: { status: 'accepted', responded_at: new Date() },
    });

    // Expire all other pending requests for this booking
    await this.prisma.driverRequest.updateMany({
      where: {
        booking_id: bookingId,
        status: 'pending',
        id: { not: driverRequest.id },
      },
      data: { status: 'cancelled' },
    });

    // Update booking
    const updatedBooking = await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        driver_id: driverId,
        booking_status: BOOKING_STATUS.DRIVER_ACCEPTED,
        accepted_time: new Date(),
      },
      include: {
        user: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            UserPhone: true,
            profile_image: true,
          },
        },
      },
    });

    // Update driver status to busy
    await this.prisma.driver.update({
      where: { id: driverId },
      data: { free_busy: 1 }, // 1 = busy
    });

    // TODO: Notify user via push notification and WebSocket

    return {
      message: 'Booking accepted',
      data: {
        ...updatedBooking,
        status_label: getStatusLabel(updatedBooking.booking_status),
      },
    };
  }

  async reject(bookingId: number, driverId: number, reason?: string) {
    // Update driver request
    await this.prisma.driverRequest.updateMany({
      where: {
        booking_id: bookingId,
        driver_id: driverId,
        status: 'pending',
      },
      data: {
        status: 'rejected',
        responded_at: new Date(),
        reject_reason: reason,
      },
    });

    return {
      message: 'Booking rejected',
    };
  }

  async onTheWay(bookingId: number, driverId: number) {
    const booking = await this.validateDriverBooking(bookingId, driverId);

    if (booking.booking_status !== BOOKING_STATUS.DRIVER_ACCEPTED) {
      throw new BadRequestException('Invalid booking status');
    }

    await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        booking_status: BOOKING_STATUS.DRIVER_ON_THE_WAY,
      },
    });

    // TODO: Notify user

    return {
      message: 'On the way to pickup',
      data: { status_label: getStatusLabel(BOOKING_STATUS.DRIVER_ON_THE_WAY) },
    };
  }

  async arrived(bookingId: number, driverId: number) {
    const booking = await this.validateDriverBooking(bookingId, driverId);

    const validStatuses = [
      BOOKING_STATUS.DRIVER_ACCEPTED,
      BOOKING_STATUS.DRIVER_ON_THE_WAY,
    ];

    if (!validStatuses.includes(booking.booking_status)) {
      throw new BadRequestException('Invalid booking status');
    }

    await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        booking_status: BOOKING_STATUS.DRIVER_ARRIVED,
        arrived_time: new Date(),
      },
    });

    // TODO: Notify user

    return {
      message: 'Arrived at pickup location',
      data: { status_label: getStatusLabel(BOOKING_STATUS.DRIVER_ARRIVED) },
    };
  }

  async start(bookingId: number, driverId: number, otp?: string) {
    const booking = await this.validateDriverBooking(bookingId, driverId);

    if (booking.booking_status !== BOOKING_STATUS.DRIVER_ARRIVED) {
      throw new BadRequestException('Invalid booking status');
    }

    // TODO: Verify OTP if required by merchant settings
    // if (merchant.require_ride_otp && booking.ride_otp !== otp) {
    //   throw new BadRequestException('Invalid OTP');
    // }

    await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        booking_status: BOOKING_STATUS.RIDE_STARTED,
        started_time: new Date(),
      },
    });

    // TODO: Start tracking, notify user

    return {
      message: 'Ride started',
      data: { status_label: getStatusLabel(BOOKING_STATUS.RIDE_STARTED) },
    };
  }

  async complete(bookingId: number, driverId: number) {
    const booking = await this.validateDriverBooking(bookingId, driverId);

    if (booking.booking_status !== BOOKING_STATUS.RIDE_STARTED) {
      throw new BadRequestException('Invalid booking status');
    }

    // Recalculate final amount if actual route differs from estimate
    // For now, use estimate amount
    const finalAmount = booking.estimate_amount || 0;

    await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        booking_status: BOOKING_STATUS.RIDE_COMPLETED,
        completed_time: new Date(),
        final_amount: finalAmount,
      },
    });

    // Update driver status to free
    await this.prisma.driver.update({
      where: { id: driverId },
      data: { free_busy: 2 }, // 2 = free
    });

    // TODO: Process payment based on payment method
    // TODO: Send invoice, notify user

    return {
      message: 'Ride completed',
      data: {
        final_amount: finalAmount,
        status_label: getStatusLabel(BOOKING_STATUS.RIDE_COMPLETED),
      },
    };
  }

  // =============================================================================
  // HELPERS
  // =============================================================================

  private async validateDriverBooking(bookingId: number, driverId: number) {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    if (booking.driver_id !== driverId) {
      throw new ForbiddenException('Access denied');
    }

    return booking;
  }

  private async getSurgeMultiplier(
    merchantId: number,
    latitude: number,
    longitude: number,
  ): Promise<number> {
    // TODO: Implement proper surge pricing based on:
    // - Time of day (peak hours)
    // - Weather conditions
    // - Supply/demand ratio
    // - Special events

    // For now, return 1.0 (no surge)
    return 1.0;
  }

  private calculateDistance(
    lat1: number,
    lon1: number,
    lat2: number,
    lon2: number,
  ): number {
    // Haversine formula
    const R = 6371; // Earth's radius in km
    const dLat = this.toRad(lat2 - lat1);
    const dLon = this.toRad(lon2 - lon1);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) *
        Math.cos(this.toRad(lat2)) *
        Math.sin(dLon / 2) *
        Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }

  private toRad(deg: number): number {
    return deg * (Math.PI / 180);
  }
}
