// =============================================================================
// BookingService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { BadRequestException, NotFoundException } from '@nestjs/common';
import { BookingService } from './booking.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  createMockUser,
  createMockDriver,
  createMockBooking,
  createMockVehicleType,
  createMockPromoCode,
  createMockZone,
  resetAllMocks,
} from '../../../test/setup';

describe('BookingService', () => {
  let service: BookingService;
  let prisma: typeof mockPrismaService;

  beforeEach(async () => {
    resetAllMocks();

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        BookingService,
        { provide: PrismaService, useValue: mockPrismaService },
      ],
    }).compile();

    service = module.get<BookingService>(BookingService);
    prisma = mockPrismaService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  // ===========================================================================
  // ESTIMATE
  // ===========================================================================

  describe('getEstimate', () => {
    it('should calculate estimate for all vehicle types', async () => {
      const vehicleTypes = [
        createMockVehicleType({ id: 1, name: 'Moto' }),
        createMockVehicleType({ id: 2, name: 'Economique' }),
      ];
      prisma.vehicleType.findMany.mockResolvedValue(vehicleTypes);
      prisma.zone.findMany.mockResolvedValue([]);

      const result = await service.getEstimate(1, {
        pickup: { latitude: 6.1319, longitude: 1.2228 },
        dropoff: { latitude: 6.1654, longitude: 1.2544 },
      });

      expect(result).toHaveProperty('estimates');
      expect(result.estimates).toHaveLength(2);
      expect(result.estimates[0]).toHaveProperty('vehicleTypeId');
      expect(result.estimates[0]).toHaveProperty('estimatedFare');
      expect(result.estimates[0]).toHaveProperty('distance');
      expect(result.estimates[0]).toHaveProperty('duration');
    });

    it('should apply surge pricing when in surge zone', async () => {
      const vehicleTypes = [createMockVehicleType()];
      const surgeZone = createMockZone({
        type: 'surge_zone',
        surge_multiplier: 1.5,
      });

      prisma.vehicleType.findMany.mockResolvedValue(vehicleTypes);
      prisma.zone.findMany.mockResolvedValue([surgeZone]);

      const result = await service.getEstimate(1, {
        pickup: { latitude: 6.1319, longitude: 1.2228 },
        dropoff: { latitude: 6.1654, longitude: 1.2544 },
      });

      expect(result.estimates[0].surgeMultiplier).toBe(1.5);
    });

    it('should apply promo code discount', async () => {
      const vehicleTypes = [createMockVehicleType()];
      const promoCode = createMockPromoCode({
        type: 'percentage',
        discount_value: 20,
        max_discount: 2000,
      });

      prisma.vehicleType.findMany.mockResolvedValue(vehicleTypes);
      prisma.zone.findMany.mockResolvedValue([]);
      prisma.promoCode.findFirst.mockResolvedValue(promoCode);

      const result = await service.getEstimate(1, {
        pickup: { latitude: 6.1319, longitude: 1.2228 },
        dropoff: { latitude: 6.1654, longitude: 1.2544 },
        promoCode: 'TEST20',
      });

      expect(result.estimates[0]).toHaveProperty('discountAmount');
      expect(result.estimates[0].discountAmount).toBeGreaterThan(0);
    });
  });

  // ===========================================================================
  // CREATE BOOKING
  // ===========================================================================

  describe('createBooking', () => {
    it('should create booking successfully', async () => {
      const mockUser = createMockUser();
      const mockVehicleType = createMockVehicleType();
      const mockBooking = createMockBooking();

      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.vehicleType.findUnique.mockResolvedValue(mockVehicleType);
      prisma.zone.findMany.mockResolvedValue([]);
      prisma.booking.create.mockResolvedValue(mockBooking);

      const result = await service.createBooking(1, 1, {
        vehicleTypeId: 1,
        pickup: { latitude: 6.1319, longitude: 1.2228, address: 'Pickup' },
        dropoff: { latitude: 6.1654, longitude: 1.2544, address: 'Dropoff' },
        paymentMethod: 'cash',
      });

      expect(result).toHaveProperty('id');
      expect(result).toHaveProperty('bookingNumber');
      expect(prisma.booking.create).toHaveBeenCalled();
    });

    it('should throw error for invalid vehicle type', async () => {
      prisma.vehicleType.findUnique.mockResolvedValue(null);

      await expect(
        service.createBooking(1, 1, {
          vehicleTypeId: 999,
          pickup: { latitude: 6.1319, longitude: 1.2228 },
          dropoff: { latitude: 6.1654, longitude: 1.2544 },
        }),
      ).rejects.toThrow(BadRequestException);
    });

    it('should throw error when pickup is in restricted zone', async () => {
      const restrictedZone = createMockZone({ type: 'restricted' });

      prisma.vehicleType.findUnique.mockResolvedValue(createMockVehicleType());
      prisma.zone.findMany.mockResolvedValue([restrictedZone]);

      // Mock point-in-polygon to return true (inside restricted zone)
      jest.spyOn(service as any, 'isPointInZone').mockReturnValue(true);

      await expect(
        service.createBooking(1, 1, {
          vehicleTypeId: 1,
          pickup: { latitude: 6.1319, longitude: 1.2228 },
          dropoff: { latitude: 6.1654, longitude: 1.2544 },
        }),
      ).rejects.toThrow(BadRequestException);
    });

    it('should apply promo code when provided', async () => {
      const mockUser = createMockUser();
      const mockVehicleType = createMockVehicleType();
      const mockPromoCode = createMockPromoCode();

      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.vehicleType.findUnique.mockResolvedValue(mockVehicleType);
      prisma.zone.findMany.mockResolvedValue([]);
      prisma.promoCode.findFirst.mockResolvedValue(mockPromoCode);
      prisma.booking.create.mockResolvedValue(
        createMockBooking({ promo_code_id: mockPromoCode.id }),
      );

      const result = await service.createBooking(1, 1, {
        vehicleTypeId: 1,
        pickup: { latitude: 6.1319, longitude: 1.2228 },
        dropoff: { latitude: 6.1654, longitude: 1.2544 },
        promoCode: 'TEST20',
      });

      expect(result).toBeDefined();
    });

    it('should schedule booking for future time', async () => {
      const scheduledTime = new Date(Date.now() + 24 * 60 * 60 * 1000); // Tomorrow

      prisma.user.findUnique.mockResolvedValue(createMockUser());
      prisma.vehicleType.findUnique.mockResolvedValue(createMockVehicleType());
      prisma.zone.findMany.mockResolvedValue([]);
      prisma.booking.create.mockResolvedValue(
        createMockBooking({ is_scheduled: true, scheduled_at: scheduledTime }),
      );

      const result = await service.createBooking(1, 1, {
        vehicleTypeId: 1,
        pickup: { latitude: 6.1319, longitude: 1.2228 },
        dropoff: { latitude: 6.1654, longitude: 1.2544 },
        scheduledAt: scheduledTime,
      });

      expect(result.is_scheduled).toBe(true);
    });
  });

  // ===========================================================================
  // GET BOOKING
  // ===========================================================================

  describe('getBooking', () => {
    it('should return booking with relations', async () => {
      const mockBooking = createMockBooking();
      prisma.booking.findFirst.mockResolvedValue({
        ...mockBooking,
        user: createMockUser(),
        driver: null,
        vehicleType: createMockVehicleType(),
      });

      const result = await service.getBooking(1, 1);

      expect(result).toHaveProperty('id');
      expect(result).toHaveProperty('user');
      expect(result).toHaveProperty('vehicleType');
    });

    it('should return null when booking not found', async () => {
      prisma.booking.findFirst.mockResolvedValue(null);

      const result = await service.getBooking(999, 1);

      expect(result).toBeNull();
    });
  });

  // ===========================================================================
  // CANCEL BOOKING
  // ===========================================================================

  describe('cancelBooking', () => {
    it('should cancel pending booking', async () => {
      const mockBooking = createMockBooking({ booking_status: 'pending' });
      prisma.booking.findFirst.mockResolvedValue(mockBooking);
      prisma.booking.update.mockResolvedValue({
        ...mockBooking,
        booking_status: 'cancelled',
      });

      const result = await service.cancelBooking(1, 1, 1, 'user', 'Changed plans');

      expect(result.booking_status).toBe('cancelled');
      expect(prisma.booking.update).toHaveBeenCalledWith(
        expect.objectContaining({
          data: expect.objectContaining({
            booking_status: 'cancelled',
            cancelled_by: 'user',
          }),
        }),
      );
    });

    it('should apply cancellation fee for started booking', async () => {
      const mockBooking = createMockBooking({
        booking_status: 'started',
        driver_id: 1,
      });
      const mockVehicleType = createMockVehicleType({ cancellation_fee: 500 });

      prisma.booking.findFirst.mockResolvedValue({
        ...mockBooking,
        vehicleType: mockVehicleType,
      });
      prisma.booking.update.mockResolvedValue({
        ...mockBooking,
        booking_status: 'cancelled',
      });

      const result = await service.cancelBooking(1, 1, 1, 'user', 'Emergency');

      expect(result).toBeDefined();
      // Should have applied cancellation fee logic
    });

    it('should throw error for completed booking', async () => {
      const mockBooking = createMockBooking({ booking_status: 'completed' });
      prisma.booking.findFirst.mockResolvedValue(mockBooking);

      await expect(
        service.cancelBooking(1, 1, 1, 'user', 'Test'),
      ).rejects.toThrow(BadRequestException);
    });
  });

  // ===========================================================================
  // DRIVER ACTIONS
  // ===========================================================================

  describe('acceptBooking', () => {
    it('should allow driver to accept searching booking', async () => {
      const mockBooking = createMockBooking({ booking_status: 'searching' });
      const mockDriver = createMockDriver();

      prisma.booking.findFirst.mockResolvedValue(mockBooking);
      prisma.driver.findUnique.mockResolvedValue(mockDriver);
      prisma.booking.update.mockResolvedValue({
        ...mockBooking,
        booking_status: 'accepted',
        driver_id: mockDriver.id,
      });
      prisma.driver.update.mockResolvedValue({ ...mockDriver, free_busy: 1 });

      const result = await service.acceptBooking(1, 1, 1);

      expect(result.booking_status).toBe('accepted');
      expect(result.driver_id).toBe(mockDriver.id);
    });

    it('should throw error when driver is not online', async () => {
      const mockBooking = createMockBooking({ booking_status: 'searching' });
      const mockDriver = createMockDriver({ is_online: 2 }); // offline

      prisma.booking.findFirst.mockResolvedValue(mockBooking);
      prisma.driver.findUnique.mockResolvedValue(mockDriver);

      await expect(
        service.acceptBooking(1, 1, 1),
      ).rejects.toThrow(BadRequestException);
    });

    it('should throw error when driver is busy', async () => {
      const mockBooking = createMockBooking({ booking_status: 'searching' });
      const mockDriver = createMockDriver({ free_busy: 1 }); // busy

      prisma.booking.findFirst.mockResolvedValue(mockBooking);
      prisma.driver.findUnique.mockResolvedValue(mockDriver);

      await expect(
        service.acceptBooking(1, 1, 1),
      ).rejects.toThrow(BadRequestException);
    });
  });

  describe('updateBookingStatus', () => {
    const statusTransitions = [
      { from: 'accepted', to: 'arrived' },
      { from: 'arrived', to: 'started' },
      { from: 'started', to: 'completed' },
    ];

    statusTransitions.forEach(({ from, to }) => {
      it(`should transition from ${from} to ${to}`, async () => {
        const mockBooking = createMockBooking({
          booking_status: from,
          driver_id: 1,
        });

        prisma.booking.findFirst.mockResolvedValue(mockBooking);
        prisma.booking.update.mockResolvedValue({
          ...mockBooking,
          booking_status: to,
        });

        const result = await service.updateBookingStatus(1, 1, 1, to);

        expect(result.booking_status).toBe(to);
      });
    });

    it('should calculate final amount on completion', async () => {
      const mockBooking = createMockBooking({
        booking_status: 'started',
        driver_id: 1,
        estimate_amount: 5000,
      });

      prisma.booking.findFirst.mockResolvedValue(mockBooking);
      prisma.booking.update.mockResolvedValue({
        ...mockBooking,
        booking_status: 'completed',
        final_amount: 5500,
      });
      prisma.driver.update.mockResolvedValue(createMockDriver());

      const result = await service.updateBookingStatus(1, 1, 1, 'completed');

      expect(result.final_amount).toBeDefined();
    });
  });

  // ===========================================================================
  // USER BOOKINGS
  // ===========================================================================

  describe('getUserBookings', () => {
    it('should return paginated user bookings', async () => {
      const mockBookings = [
        createMockBooking({ id: 1 }),
        createMockBooking({ id: 2 }),
      ];

      prisma.booking.findMany.mockResolvedValue(mockBookings);
      prisma.booking.count.mockResolvedValue(2);

      const result = await service.getUserBookings(1, 1, {
        page: 1,
        limit: 10,
      });

      expect(result).toHaveProperty('data');
      expect(result).toHaveProperty('total');
      expect(result).toHaveProperty('page');
      expect(result.data).toHaveLength(2);
    });

    it('should filter by status', async () => {
      prisma.booking.findMany.mockResolvedValue([
        createMockBooking({ booking_status: 'completed' }),
      ]);
      prisma.booking.count.mockResolvedValue(1);

      const result = await service.getUserBookings(1, 1, {
        status: 'completed',
      });

      expect(result.data[0].booking_status).toBe('completed');
    });
  });

  // ===========================================================================
  // DRIVER BOOKINGS
  // ===========================================================================

  describe('getDriverBookings', () => {
    it('should return paginated driver bookings', async () => {
      const mockBookings = [
        createMockBooking({ id: 1, driver_id: 1 }),
        createMockBooking({ id: 2, driver_id: 1 }),
      ];

      prisma.booking.findMany.mockResolvedValue(mockBookings);
      prisma.booking.count.mockResolvedValue(2);

      const result = await service.getDriverBookings(1, 1, {
        page: 1,
        limit: 10,
      });

      expect(result.data).toHaveLength(2);
    });
  });

  // ===========================================================================
  // ACTIVE BOOKING
  // ===========================================================================

  describe('getActiveBooking', () => {
    it('should return active user booking', async () => {
      const mockBooking = createMockBooking({ booking_status: 'started' });
      prisma.booking.findFirst.mockResolvedValue(mockBooking);

      const result = await service.getActiveUserBooking(1, 1);

      expect(result).toBeDefined();
      expect(['pending', 'searching', 'accepted', 'arrived', 'started']).toContain(
        result.booking_status,
      );
    });

    it('should return null when no active booking', async () => {
      prisma.booking.findFirst.mockResolvedValue(null);

      const result = await service.getActiveUserBooking(1, 1);

      expect(result).toBeNull();
    });

    it('should return active driver booking', async () => {
      const mockBooking = createMockBooking({
        booking_status: 'accepted',
        driver_id: 1,
      });
      prisma.booking.findFirst.mockResolvedValue(mockBooking);

      const result = await service.getActiveDriverBooking(1, 1);

      expect(result).toBeDefined();
    });
  });

  // ===========================================================================
  // HELPER METHODS
  // ===========================================================================

  describe('calculateDistance', () => {
    it('should calculate distance between two points', () => {
      // Lome Centre to Lome Airport (approx 5-6 km)
      const distance = (service as any).calculateDistance(
        6.1319, 1.2228, // Lome Centre
        6.1654, 1.2544, // Lome Airport
      );

      expect(distance).toBeGreaterThan(4);
      expect(distance).toBeLessThan(8);
    });

    it('should return 0 for same point', () => {
      const distance = (service as any).calculateDistance(
        6.1319, 1.2228,
        6.1319, 1.2228,
      );

      expect(distance).toBe(0);
    });
  });

  describe('generateBookingNumber', () => {
    it('should generate unique booking number', async () => {
      prisma.booking.findFirst.mockResolvedValue(null);

      const bookingNumber = await (service as any).generateBookingNumber(1);

      expect(bookingNumber).toMatch(/^BK-\d{6}-\d{5}$/);
    });
  });
});
