import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';

import { BookingService } from './booking.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentUser, MerchantId } from '../../common/decorators/merchant.decorator';

import { CreateBookingDto } from './dto/create-booking.dto';
import { EstimateBookingDto } from './dto/estimate-booking.dto';
import { RateBookingDto } from './dto/rate-booking.dto';
import { CancelBookingDto } from './dto/cancel-booking.dto';

@ApiTags('bookings')
@Controller('booking')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class BookingController {
  constructor(private readonly bookingService: BookingService) {}

  // ============================================================================
  // USER ENDPOINTS
  // ============================================================================

  @Post('estimate')
  @ApiOperation({ summary: 'Get fare estimate for a ride' })
  async estimate(
    @Body() dto: EstimateBookingDto,
    @MerchantId() merchantId: number,
  ) {
    return this.bookingService.estimate(dto, merchantId);
  }

  @Post('create')
  @ApiOperation({ summary: 'Create a new booking' })
  async create(
    @CurrentUser() user: any,
    @Body() dto: CreateBookingDto,
    @MerchantId() merchantId: number,
  ) {
    return this.bookingService.create(user.id, dto, merchantId);
  }

  @Get(':id')
  @ApiOperation({ summary: 'Get booking details' })
  async getBooking(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.bookingService.getBooking(id, user.id, user.type);
  }

  @Post(':id/cancel')
  @ApiOperation({ summary: 'Cancel a booking' })
  async cancel(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
    @Body() dto: CancelBookingDto,
  ) {
    return this.bookingService.cancel(id, user.id, user.type, dto);
  }

  @Post(':id/rate')
  @ApiOperation({ summary: 'Rate a completed booking' })
  async rate(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
    @Body() dto: RateBookingDto,
  ) {
    return this.bookingService.rate(id, user.id, user.type, dto);
  }

  // ============================================================================
  // DRIVER ENDPOINTS
  // ============================================================================

  @Post(':id/accept')
  @ApiOperation({ summary: 'Driver accepts a booking' })
  async accept(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.bookingService.accept(id, user.id);
  }

  @Post(':id/reject')
  @ApiOperation({ summary: 'Driver rejects a booking' })
  async reject(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.bookingService.reject(id, user.id);
  }

  @Post(':id/arrived')
  @ApiOperation({ summary: 'Driver arrived at pickup' })
  async arrived(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.bookingService.arrived(id, user.id);
  }

  @Post(':id/start')
  @ApiOperation({ summary: 'Driver starts the ride' })
  async start(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.bookingService.start(id, user.id);
  }

  @Post(':id/complete')
  @ApiOperation({ summary: 'Driver completes the ride' })
  async complete(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.bookingService.complete(id, user.id);
  }
}
