import { Injectable, UnauthorizedException } from '@nestjs/common';
import { PassportStrategy } from '@nestjs/passport';
import { ExtractJwt, Strategy } from 'passport-jwt';
import { ConfigService } from '@nestjs/config';
import { PrismaService } from '../../../common/prisma/prisma.service';

interface JwtPayload {
  sub: number;
  type: 'user' | 'driver';
  merchantId: number;
}

@Injectable()
export class JwtStrategy extends PassportStrategy(Strategy) {
  constructor(
    private configService: ConfigService,
    private prisma: PrismaService,
  ) {
    super({
      jwtFromRequest: ExtractJwt.fromAuthHeaderAsBearerToken(),
      ignoreExpiration: false,
      secretOrKey: configService.get<string>('jwt.secret'),
    });
  }

  async validate(payload: JwtPayload) {
    const { sub: id, type, merchantId } = payload;

    let entity: any;

    if (type === 'user') {
      entity = await this.prisma.user.findFirst({
        where: {
          id,
          merchant_id: merchantId,
          user_delete: null,
          UserStatus: 1,
        },
      });
    } else if (type === 'driver') {
      entity = await this.prisma.driver.findFirst({
        where: {
          id,
          merchant_id: merchantId,
          driver_status: 1,
        },
      });
    }

    if (!entity) {
      throw new UnauthorizedException('Invalid token');
    }

    // Remove sensitive fields
    const { password, otp, otp_expires_at, ...sanitized } = entity;

    return {
      ...sanitized,
      type,
      merchantId,
    };
  }
}
