import {
  Controller,
  Post,
  Body,
  UseGuards,
  HttpCode,
  HttpStatus,
  Get,
  Request,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';

import { AuthService } from './auth.service';
import { Public } from '../../common/decorators/public.decorator';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

// DTOs
import { UserSignupDto } from './dto/user-signup.dto';
import { UserLoginDto } from './dto/user-login.dto';
import { DriverSignupDto } from './dto/driver-signup.dto';
import { DriverLoginDto } from './dto/driver-login.dto';
import { OtpDto } from './dto/otp.dto';
import { ForgotPasswordDto } from './dto/forgot-password.dto';
import { ResetPasswordDto } from './dto/reset-password.dto';
import { RefreshTokenDto } from './dto/refresh-token.dto';

@ApiTags('auth')
@Controller('auth')
@UseGuards(MerchantGuard)
export class AuthController {
  constructor(private readonly authService: AuthService) {}

  // ============================================================================
  // USER AUTHENTICATION
  // ============================================================================

  @Public()
  @Post('user/signup')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'User signup' })
  async userSignup(
    @Body() dto: UserSignupDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.userSignup(dto, merchantId);
  }

  @Public()
  @Post('user/login')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'User login with email/phone and password' })
  async userLogin(
    @Body() dto: UserLoginDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.userLogin(dto, merchantId);
  }

  @Public()
  @Post('user/login-otp')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'User login with OTP' })
  async userLoginOtp(
    @Body() dto: OtpDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.userLoginOtp(dto, merchantId);
  }

  @Public()
  @Post('user/forgot-password')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Send password reset OTP to user' })
  async userForgotPassword(
    @Body() dto: ForgotPasswordDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.userForgotPassword(dto, merchantId);
  }

  @Public()
  @Post('user/reset-password')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Reset user password with OTP' })
  async userResetPassword(
    @Body() dto: ResetPasswordDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.userResetPassword(dto, merchantId);
  }

  @Post('user/logout')
  @HttpCode(HttpStatus.OK)
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  @ApiOperation({ summary: 'User logout' })
  async userLogout(@CurrentUser() user: any) {
    return this.authService.userLogout(user.id);
  }

  // ============================================================================
  // DRIVER AUTHENTICATION
  // ============================================================================

  @Public()
  @Post('driver/signup')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Driver signup' })
  async driverSignup(
    @Body() dto: DriverSignupDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.driverSignup(dto, merchantId);
  }

  @Public()
  @Post('driver/login')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Driver login with email/phone and password' })
  async driverLogin(
    @Body() dto: DriverLoginDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.driverLogin(dto, merchantId);
  }

  @Public()
  @Post('driver/login-otp')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Driver login with OTP' })
  async driverLoginOtp(
    @Body() dto: OtpDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.driverLoginOtp(dto, merchantId);
  }

  @Public()
  @Post('driver/forgot-password')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Send password reset OTP to driver' })
  async driverForgotPassword(
    @Body() dto: ForgotPasswordDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.driverForgotPassword(dto, merchantId);
  }

  @Public()
  @Post('driver/reset-password')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Reset driver password with OTP' })
  async driverResetPassword(
    @Body() dto: ResetPasswordDto,
    @MerchantId() merchantId: number,
  ) {
    return this.authService.driverResetPassword(dto, merchantId);
  }

  @Post('driver/logout')
  @HttpCode(HttpStatus.OK)
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  @ApiOperation({ summary: 'Driver logout' })
  async driverLogout(@CurrentUser() user: any) {
    return this.authService.driverLogout(user.id);
  }

  // ============================================================================
  // COMMON
  // ============================================================================

  @Public()
  @Post('refresh-token')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Refresh access token' })
  async refreshToken(@Body() dto: RefreshTokenDto) {
    return this.authService.refreshToken(dto.refresh_token);
  }

  @Get('me')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  @ApiOperation({ summary: 'Get current authenticated user' })
  async me(@CurrentUser() user: any) {
    return {
      message: 'User profile',
      data: user,
    };
  }
}
