import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiKeyService, ApiKeyPermission } from './api-key.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class CreateApiKeyDto {
  name: string;
  permissions: ApiKeyPermission[];
  rateLimit?: number;
  rateLimitWindow?: number;
  ipWhitelist?: string[];
  expiresAt?: string;
}

class UpdateApiKeyDto {
  name?: string;
  permissions?: ApiKeyPermission[];
  rateLimit?: number;
  rateLimitWindow?: number;
  ipWhitelist?: string[];
  expiresAt?: string | null;
  active?: boolean;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('api-keys')
@UseGuards(JwtAuthGuard)
export class ApiKeyController {
  constructor(private readonly apiKeyService: ApiKeyService) {}

  /**
   * List API keys
   */
  @Get()
  async listKeys(@MerchantId() merchantId: number) {
    const keys = await this.apiKeyService.listKeys(merchantId);

    // Hide sensitive data
    return keys.map((key) => ({
      id: key.id,
      name: key.name,
      keyPrefix: key.keyPrefix + '...',
      permissions: key.permissions,
      rateLimit: key.rateLimit,
      rateLimitWindow: key.rateLimitWindow,
      ipWhitelist: key.ipWhitelist,
      expiresAt: key.expiresAt,
      lastUsedAt: key.lastUsedAt,
      usageCount: key.usageCount,
      active: key.active,
      createdAt: key.createdAt,
    }));
  }

  /**
   * Create API key
   */
  @Post()
  async createKey(
    @MerchantId() merchantId: number,
    @Body() dto: CreateApiKeyDto,
  ) {
    const result = await this.apiKeyService.createKey(merchantId, {
      name: dto.name,
      permissions: dto.permissions,
      rateLimit: dto.rateLimit,
      rateLimitWindow: dto.rateLimitWindow,
      ipWhitelist: dto.ipWhitelist,
      expiresAt: dto.expiresAt ? new Date(dto.expiresAt) : undefined,
    });

    return {
      success: true,
      apiKey: {
        id: result.apiKey.id,
        name: result.apiKey.name,
        keyPrefix: result.apiKey.keyPrefix + '...',
        permissions: result.apiKey.permissions,
        rateLimit: result.apiKey.rateLimit,
        active: result.apiKey.active,
        createdAt: result.apiKey.createdAt,
      },
      rawKey: result.rawKey,
      message: 'Save the API key securely. It will not be shown again.',
    };
  }

  /**
   * Get API key details
   */
  @Get(':id')
  async getKey(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const key = await this.apiKeyService.getKey(id, merchantId);

    if (!key) {
      return { error: 'API key not found' };
    }

    return {
      id: key.id,
      name: key.name,
      keyPrefix: key.keyPrefix + '...',
      permissions: key.permissions,
      rateLimit: key.rateLimit,
      rateLimitWindow: key.rateLimitWindow,
      ipWhitelist: key.ipWhitelist,
      expiresAt: key.expiresAt,
      lastUsedAt: key.lastUsedAt,
      usageCount: key.usageCount,
      active: key.active,
      createdAt: key.createdAt,
      updatedAt: key.updatedAt,
    };
  }

  /**
   * Update API key
   */
  @Put(':id')
  async updateKey(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateApiKeyDto,
  ) {
    const key = await this.apiKeyService.updateKey(id, merchantId, {
      name: dto.name,
      permissions: dto.permissions,
      rateLimit: dto.rateLimit,
      rateLimitWindow: dto.rateLimitWindow,
      ipWhitelist: dto.ipWhitelist,
      expiresAt: dto.expiresAt === null ? null : dto.expiresAt ? new Date(dto.expiresAt) : undefined,
      active: dto.active,
    });

    if (!key) {
      return { error: 'API key not found' };
    }

    return {
      success: true,
      apiKey: {
        id: key.id,
        name: key.name,
        permissions: key.permissions,
        rateLimit: key.rateLimit,
        active: key.active,
        updatedAt: key.updatedAt,
      },
    };
  }

  /**
   * Revoke API key
   */
  @Post(':id/revoke')
  async revokeKey(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const revoked = await this.apiKeyService.revokeKey(id, merchantId);

    return {
      success: revoked,
      message: revoked ? 'API key revoked' : 'API key not found',
    };
  }

  /**
   * Delete API key
   */
  @Delete(':id')
  async deleteKey(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.apiKeyService.deleteKey(id, merchantId);

    return {
      success: deleted,
      message: deleted ? 'API key deleted' : 'API key not found',
    };
  }

  /**
   * Regenerate API key
   */
  @Post(':id/regenerate')
  async regenerateKey(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const result = await this.apiKeyService.regenerateKey(id, merchantId);

    if (!result) {
      return { error: 'API key not found' };
    }

    return {
      success: true,
      rawKey: result.rawKey,
      message: 'Save the new API key securely. It will not be shown again.',
    };
  }

  /**
   * Get API key usage statistics
   */
  @Get(':id/usage')
  async getUsageStats(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.apiKeyService.getUsageStats(
      id,
      merchantId,
      startDate ? new Date(startDate) : undefined,
      endDate ? new Date(endDate) : undefined,
    );
  }

  /**
   * Get available permissions
   */
  @Get('meta/permissions')
  getPermissions() {
    return {
      permissions: [
        { permission: 'read:bookings', description: 'Read booking data' },
        { permission: 'write:bookings', description: 'Create and update bookings' },
        { permission: 'read:drivers', description: 'Read driver data' },
        { permission: 'write:drivers', description: 'Update driver data' },
        { permission: 'read:users', description: 'Read user data' },
        { permission: 'write:users', description: 'Update user data' },
        { permission: 'read:payments', description: 'Read payment data' },
        { permission: 'write:payments', description: 'Process payments' },
        { permission: 'read:analytics', description: 'Access analytics data' },
        { permission: 'read:settings', description: 'Read settings' },
        { permission: 'write:settings', description: 'Update settings' },
        { permission: 'webhooks', description: 'Manage webhooks' },
        { permission: '*', description: 'Full access (all permissions)' },
      ],
    };
  }

  /**
   * Get API documentation
   */
  @Get('meta/docs')
  getDocs() {
    return {
      authentication: {
        header: 'Authorization: Bearer mk_your_api_key',
        alternative: 'X-API-Key: mk_your_api_key',
      },
      rateLimit: {
        headers: {
          'X-RateLimit-Limit': 'Maximum requests allowed in window',
          'X-RateLimit-Remaining': 'Remaining requests in current window',
          'X-RateLimit-Reset': 'Unix timestamp when window resets',
        },
      },
      errors: {
        401: 'Unauthorized - Invalid or missing API key',
        403: 'Forbidden - IP not allowed, rate limit exceeded, or insufficient permissions',
        429: 'Too Many Requests - Rate limit exceeded',
      },
    };
  }
}
