import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface RealtimeStats {
  activeBookings: number;
  pendingBookings: number;
  onlineDrivers: number;
  availableDrivers: number;
  todayBookings: number;
  todayRevenue: number;
  todayCompletedBookings: number;
  todayCancelledBookings: number;
}

export interface LiveBooking {
  id: number;
  bookingNumber: string;
  status: string;
  userName: string;
  driverName: string | null;
  pickupAddress: string;
  dropoffAddress: string;
  createdAt: Date;
}

export interface LiveDriver {
  id: number;
  name: string;
  lat: number;
  lng: number;
  isAvailable: boolean;
  currentBookingId: number | null;
  rating: number;
  vehicleType: string;
}

@Injectable()
export class RealtimeAnalyticsService {
  private readonly logger = new Logger(RealtimeAnalyticsService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Get real-time dashboard stats
   */
  async getRealtimeStats(merchantId: number): Promise<RealtimeStats> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const [
      activeBookings,
      pendingBookings,
      onlineDrivers,
      availableDrivers,
      todayStats,
    ] = await Promise.all([
      // Active bookings (in progress)
      this.prisma.booking.count({
        where: {
          merchant_id: merchantId,
          status: { in: ['accepted', 'arrived', 'in_progress'] },
        },
      }),

      // Pending bookings
      this.prisma.booking.count({
        where: {
          merchant_id: merchantId,
          status: 'pending',
        },
      }),

      // Online drivers
      this.prisma.driver.count({
        where: {
          merchant_id: merchantId,
          is_online: true,
        },
      }),

      // Available drivers (online and not on a booking)
      this.prisma.driver.count({
        where: {
          merchant_id: merchantId,
          is_online: true,
          is_available: true,
        },
      }),

      // Today's stats
      this.prisma.booking.aggregate({
        where: {
          merchant_id: merchantId,
          created_at: { gte: today },
        },
        _count: { id: true },
        _sum: { total_fare: true },
      }),
    ]);

    // Today's completed and cancelled
    const [todayCompleted, todayCancelled] = await Promise.all([
      this.prisma.booking.count({
        where: {
          merchant_id: merchantId,
          status: 'completed',
          ride_ended_at: { gte: today },
        },
      }),
      this.prisma.booking.count({
        where: {
          merchant_id: merchantId,
          status: 'cancelled',
          cancelled_at: { gte: today },
        },
      }),
    ]);

    return {
      activeBookings,
      pendingBookings,
      onlineDrivers,
      availableDrivers,
      todayBookings: todayStats._count.id,
      todayRevenue: todayStats._sum.total_fare || 0,
      todayCompletedBookings: todayCompleted,
      todayCancelledBookings: todayCancelled,
    };
  }

  /**
   * Get live bookings feed
   */
  async getLiveBookings(
    merchantId: number,
    limit: number = 20,
  ): Promise<LiveBooking[]> {
    const bookings = await this.prisma.booking.findMany({
      where: {
        merchant_id: merchantId,
        status: { in: ['pending', 'accepted', 'arrived', 'in_progress'] },
      },
      select: {
        id: true,
        booking_number: true,
        status: true,
        pickup_address: true,
        dropoff_address: true,
        created_at: true,
        user: { select: { name: true } },
        driver: { select: { name: true } },
      },
      orderBy: { created_at: 'desc' },
      take: limit,
    });

    return bookings.map(b => ({
      id: b.id,
      bookingNumber: b.booking_number,
      status: b.status,
      userName: b.user?.name || 'Unknown',
      driverName: b.driver?.name || null,
      pickupAddress: b.pickup_address || '',
      dropoffAddress: b.dropoff_address || '',
      createdAt: b.created_at,
    }));
  }

  /**
   * Get live driver locations
   */
  async getLiveDrivers(
    merchantId: number,
    bounds?: { minLat: number; maxLat: number; minLng: number; maxLng: number },
  ): Promise<LiveDriver[]> {
    const where: any = {
      merchant_id: merchantId,
      is_online: true,
      current_lat: { not: null },
      current_lng: { not: null },
    };

    if (bounds) {
      where.current_lat = { gte: bounds.minLat, lte: bounds.maxLat };
      where.current_lng = { gte: bounds.minLng, lte: bounds.maxLng };
    }

    const drivers = await this.prisma.driver.findMany({
      where,
      select: {
        id: true,
        name: true,
        current_lat: true,
        current_lng: true,
        is_available: true,
        current_booking_id: true,
        rating: true,
        vehicle_type: true,
      },
    });

    return drivers.map(d => ({
      id: d.id,
      name: d.name,
      lat: d.current_lat,
      lng: d.current_lng,
      isAvailable: d.is_available,
      currentBookingId: d.current_booking_id,
      rating: d.rating || 0,
      vehicleType: d.vehicle_type || 'sedan',
    }));
  }

  /**
   * Get recent activity feed
   */
  async getActivityFeed(
    merchantId: number,
    limit: number = 50,
  ): Promise<Array<{
    type: string;
    message: string;
    timestamp: Date;
    data?: Record<string, any>;
  }>> {
    const activities: Array<{
      type: string;
      message: string;
      timestamp: Date;
      data?: Record<string, any>;
    }> = [];

    // Recent bookings
    const recentBookings = await this.prisma.booking.findMany({
      where: { merchant_id: merchantId },
      orderBy: { created_at: 'desc' },
      take: 20,
      select: {
        id: true,
        booking_number: true,
        status: true,
        created_at: true,
        user: { select: { name: true } },
      },
    });

    for (const booking of recentBookings) {
      activities.push({
        type: 'booking',
        message: `Booking #${booking.booking_number} ${booking.status} by ${booking.user?.name || 'Unknown'}`,
        timestamp: booking.created_at,
        data: { bookingId: booking.id, status: booking.status },
      });
    }

    // Recent driver status changes
    const recentDriverChanges = await this.prisma.driver.findMany({
      where: {
        merchant_id: merchantId,
        went_online_at: { not: null },
      },
      orderBy: { went_online_at: 'desc' },
      take: 10,
      select: {
        id: true,
        name: true,
        is_online: true,
        went_online_at: true,
      },
    });

    for (const driver of recentDriverChanges) {
      if (driver.went_online_at) {
        activities.push({
          type: 'driver',
          message: `Driver ${driver.name} went ${driver.is_online ? 'online' : 'offline'}`,
          timestamp: driver.went_online_at,
          data: { driverId: driver.id, isOnline: driver.is_online },
        });
      }
    }

    // Sort by timestamp and limit
    return activities
      .sort((a, b) => b.timestamp.getTime() - a.timestamp.getTime())
      .slice(0, limit);
  }

  /**
   * Get current demand/supply ratio by zone
   */
  async getDemandSupplyByZone(merchantId: number): Promise<Array<{
    zoneId: number;
    zoneName: string;
    pendingBookings: number;
    availableDrivers: number;
    ratio: number;
    surgeMultiplier: number;
  }>> {
    const zones = await this.prisma.zone.findMany({
      where: { merchant_id: merchantId },
      select: {
        id: true,
        name: true,
        min_lat: true,
        max_lat: true,
        min_lng: true,
        max_lng: true,
        surge_multiplier: true,
      },
    });

    const results = await Promise.all(
      zones.map(async (zone) => {
        const [pendingBookings, availableDrivers] = await Promise.all([
          this.prisma.booking.count({
            where: {
              merchant_id: merchantId,
              status: 'pending',
              pickup_lat: { gte: zone.min_lat, lte: zone.max_lat },
              pickup_lng: { gte: zone.min_lng, lte: zone.max_lng },
            },
          }),
          this.prisma.driver.count({
            where: {
              merchant_id: merchantId,
              is_online: true,
              is_available: true,
              current_lat: { gte: zone.min_lat, lte: zone.max_lat },
              current_lng: { gte: zone.min_lng, lte: zone.max_lng },
            },
          }),
        ]);

        const ratio = availableDrivers > 0
          ? Math.round((pendingBookings / availableDrivers) * 100) / 100
          : pendingBookings > 0 ? 999 : 0;

        return {
          zoneId: zone.id,
          zoneName: zone.name,
          pendingBookings,
          availableDrivers,
          ratio,
          surgeMultiplier: zone.surge_multiplier || 1,
        };
      }),
    );

    return results.sort((a, b) => b.ratio - a.ratio);
  }

  /**
   * Get booking funnel stats
   */
  async getBookingFunnel(merchantId: number): Promise<{
    requested: number;
    accepted: number;
    started: number;
    completed: number;
    cancelled: number;
    conversionRate: number;
  }> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const stats = await this.prisma.booking.groupBy({
      by: ['status'],
      where: {
        merchant_id: merchantId,
        created_at: { gte: today },
      },
      _count: { id: true },
    });

    const getCount = (status: string) =>
      stats.find(s => s.status === status)?._count.id || 0;

    const requested = stats.reduce((sum, s) => sum + s._count.id, 0);
    const completed = getCount('completed');

    return {
      requested,
      accepted: getCount('accepted') + getCount('arrived') + getCount('in_progress') + completed,
      started: getCount('in_progress') + completed,
      completed,
      cancelled: getCount('cancelled'),
      conversionRate: requested > 0
        ? Math.round((completed / requested) * 10000) / 100
        : 0,
    };
  }

  /**
   * Get average wait times
   */
  async getAverageWaitTimes(merchantId: number): Promise<{
    avgWaitTime: number;
    avgPickupTime: number;
    avgTripDuration: number;
  }> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const stats = await this.prisma.booking.aggregate({
      where: {
        merchant_id: merchantId,
        status: 'completed',
        ride_ended_at: { gte: today },
      },
      _avg: {
        wait_time: true,
        pickup_time: true,
        duration: true,
      },
    });

    return {
      avgWaitTime: Math.round((stats._avg.wait_time || 0) * 10) / 10,
      avgPickupTime: Math.round((stats._avg.pickup_time || 0) * 10) / 10,
      avgTripDuration: Math.round((stats._avg.duration || 0) * 10) / 10,
    };
  }
}
