import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface MetricValue {
  value: number;
  timestamp: Date;
  tags?: Record<string, string>;
}

@Injectable()
export class MetricsService {
  private readonly logger = new Logger(MetricsService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Track a metric
   */
  async track(name: string, value: number, tags?: Record<string, string>): Promise<void> {
    try {
      await this.prisma.metric.create({
        data: {
          name,
          value,
          tags: tags ? JSON.stringify(tags) : null,
          timestamp: new Date(),
        },
      });
    } catch (error) {
      this.logger.error(`Failed to track metric ${name}: ${error.message}`);
    }
  }

  /**
   * Increment a counter metric
   */
  async increment(name: string, amount: number = 1, tags?: Record<string, string>): Promise<void> {
    await this.track(name, amount, tags);
  }

  /**
   * Track a timing metric
   */
  async timing(name: string, durationMs: number, tags?: Record<string, string>): Promise<void> {
    await this.track(`${name}.timing`, durationMs, tags);
  }

  /**
   * Track a gauge metric (current value)
   */
  async gauge(name: string, value: number, tags?: Record<string, string>): Promise<void> {
    await this.track(`${name}.gauge`, value, tags);
  }

  /**
   * Get metric values over time
   */
  async getMetric(
    name: string,
    startDate: Date,
    endDate: Date,
    aggregation: 'sum' | 'avg' | 'min' | 'max' | 'count' = 'sum',
    granularity: 'minute' | 'hour' | 'day' = 'hour',
  ): Promise<Array<{ period: string; value: number }>> {
    const groupBy = granularity === 'minute'
      ? "DATE_FORMAT(timestamp, '%Y-%m-%d %H:%i')"
      : granularity === 'hour'
      ? "DATE_FORMAT(timestamp, '%Y-%m-%d %H:00')"
      : "DATE(timestamp)";

    const aggFn = {
      sum: 'SUM(value)',
      avg: 'AVG(value)',
      min: 'MIN(value)',
      max: 'MAX(value)',
      count: 'COUNT(*)',
    }[aggregation];

    const result = await this.prisma.$queryRawUnsafe(`
      SELECT
        ${groupBy} as period,
        ${aggFn} as value
      FROM metrics
      WHERE name = '${name}'
        AND timestamp >= '${startDate.toISOString()}'
        AND timestamp <= '${endDate.toISOString()}'
      GROUP BY ${groupBy}
      ORDER BY period
    `);

    return result as Array<{ period: string; value: number }>;
  }

  /**
   * Get current metric value
   */
  async getCurrentValue(name: string): Promise<number | null> {
    const metric = await this.prisma.metric.findFirst({
      where: { name },
      orderBy: { timestamp: 'desc' },
    });

    return metric?.value ?? null;
  }

  // ============================================================================
  // PREDEFINED METRICS
  // ============================================================================

  /**
   * Track booking created
   */
  async trackBookingCreated(merchantId: number, serviceType: string): Promise<void> {
    await this.increment('bookings.created', 1, {
      merchant_id: merchantId.toString(),
      service_type: serviceType,
    });
  }

  /**
   * Track booking completed
   */
  async trackBookingCompleted(
    merchantId: number,
    serviceType: string,
    fare: number,
    duration: number,
  ): Promise<void> {
    await this.increment('bookings.completed', 1, {
      merchant_id: merchantId.toString(),
      service_type: serviceType,
    });
    await this.track('bookings.revenue', fare, {
      merchant_id: merchantId.toString(),
      service_type: serviceType,
    });
    await this.timing('bookings.duration', duration * 1000, {
      merchant_id: merchantId.toString(),
    });
  }

  /**
   * Track booking cancelled
   */
  async trackBookingCancelled(merchantId: number, reason: string): Promise<void> {
    await this.increment('bookings.cancelled', 1, {
      merchant_id: merchantId.toString(),
      reason,
    });
  }

  /**
   * Track driver online
   */
  async trackDriverOnline(merchantId: number): Promise<void> {
    await this.increment('drivers.online', 1, {
      merchant_id: merchantId.toString(),
    });
  }

  /**
   * Track driver offline
   */
  async trackDriverOffline(merchantId: number): Promise<void> {
    await this.increment('drivers.offline', 1, {
      merchant_id: merchantId.toString(),
    });
  }

  /**
   * Track API request
   */
  async trackApiRequest(endpoint: string, method: string, status: number, duration: number): Promise<void> {
    await this.increment('api.requests', 1, {
      endpoint,
      method,
      status: status.toString(),
    });
    await this.timing('api.latency', duration, {
      endpoint,
      method,
    });
  }

  /**
   * Track payment
   */
  async trackPayment(merchantId: number, method: string, amount: number, success: boolean): Promise<void> {
    const status = success ? 'success' : 'failed';
    await this.increment(`payments.${status}`, 1, {
      merchant_id: merchantId.toString(),
      method,
    });
    if (success) {
      await this.track('payments.amount', amount, {
        merchant_id: merchantId.toString(),
        method,
      });
    }
  }

  /**
   * Track user signup
   */
  async trackUserSignup(merchantId: number, platform: string): Promise<void> {
    await this.increment('users.signup', 1, {
      merchant_id: merchantId.toString(),
      platform,
    });
  }

  /**
   * Track driver signup
   */
  async trackDriverSignup(merchantId: number): Promise<void> {
    await this.increment('drivers.signup', 1, {
      merchant_id: merchantId.toString(),
    });
  }

  /**
   * Track notification sent
   */
  async trackNotificationSent(type: string, channel: string, success: boolean): Promise<void> {
    const status = success ? 'success' : 'failed';
    await this.increment(`notifications.${status}`, 1, {
      type,
      channel,
    });
  }

  /**
   * Track SMS sent
   */
  async trackSmsSent(provider: string, success: boolean): Promise<void> {
    const status = success ? 'success' : 'failed';
    await this.increment(`sms.${status}`, 1, { provider });
  }

  // ============================================================================
  // AGGREGATED METRICS
  // ============================================================================

  /**
   * Get summary metrics for dashboard
   */
  async getDashboardMetrics(merchantId: number): Promise<{
    today: Record<string, number>;
    yesterday: Record<string, number>;
    thisWeek: Record<string, number>;
    thisMonth: Record<string, number>;
  }> {
    const now = new Date();
    const todayStart = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const yesterdayStart = new Date(todayStart);
    yesterdayStart.setDate(yesterdayStart.getDate() - 1);
    const weekStart = new Date(todayStart);
    weekStart.setDate(weekStart.getDate() - 7);
    const monthStart = new Date(now.getFullYear(), now.getMonth(), 1);

    const [today, yesterday, thisWeek, thisMonth] = await Promise.all([
      this.getMetricsSummary(merchantId, todayStart, now),
      this.getMetricsSummary(merchantId, yesterdayStart, todayStart),
      this.getMetricsSummary(merchantId, weekStart, now),
      this.getMetricsSummary(merchantId, monthStart, now),
    ]);

    return { today, yesterday, thisWeek, thisMonth };
  }

  private async getMetricsSummary(
    merchantId: number,
    startDate: Date,
    endDate: Date,
  ): Promise<Record<string, number>> {
    const metrics = await this.prisma.$queryRaw<Array<{ name: string; total: number }>>`
      SELECT
        name,
        SUM(value) as total
      FROM metrics
      WHERE timestamp >= ${startDate}
        AND timestamp < ${endDate}
        AND (tags IS NULL OR JSON_EXTRACT(tags, '$.merchant_id') = ${merchantId.toString()})
      GROUP BY name
    `;

    const result: Record<string, number> = {};
    for (const metric of metrics) {
      result[metric.name] = Number(metric.total);
    }
    return result;
  }
}
