import { NestFactory } from '@nestjs/core';
import { ValidationPipe } from '@nestjs/common';
import { SwaggerModule, DocumentBuilder } from '@nestjs/swagger';
import { AppModule } from './app.module';
import * as compression from 'compression';
import helmet from 'helmet';

async function bootstrap() {
  const app = await NestFactory.create(AppModule);

  // Security
  app.use(helmet());
  app.use(compression());

  // CORS
  app.enableCors({
    origin: process.env.FRONTEND_URL || '*',
    methods: 'GET,HEAD,PUT,PATCH,POST,DELETE,OPTIONS',
    credentials: true,
  });

  // Global validation pipe
  app.useGlobalPipes(
    new ValidationPipe({
      whitelist: true,
      forbidNonWhitelisted: true,
      transform: true,
      transformOptions: {
        enableImplicitConversion: true,
      },
    }),
  );

  // API prefix
  app.setGlobalPrefix('api');

  // Swagger documentation
  if (process.env.NODE_ENV !== 'production') {
    const config = new DocumentBuilder()
      .setTitle('MonkAPI')
      .setDescription(`
# MonkAPI - VTC Backend API

Backend complet pour application de transport (VTC, livraison, handyman).

## Authentification
Utilisez un token JWT Bearer pour les endpoints protégés.

## Multi-tenant
Toutes les requêtes sont isolées par \`merchant_id\`.

## Rate Limiting
- 100 requêtes par minute par IP
- 1000 requêtes par minute avec API Key

## Codes de réponse
- \`200\` - Succès
- \`201\` - Créé
- \`400\` - Requête invalide
- \`401\` - Non authentifié
- \`403\` - Non autorisé
- \`404\` - Non trouvé
- \`429\` - Trop de requêtes
- \`500\` - Erreur serveur
      `)
      .setVersion(process.env.API_VERSION || '1.5')
      .setContact('MonkAPI Support', 'https://monkapi.com', 'support@monkapi.com')
      .setLicense('Proprietary', 'https://monkapi.com/license')
      .addServer('http://localhost:3000', 'Development')
      .addServer('https://api.monkapi.com', 'Production')
      .addBearerAuth(
        {
          type: 'http',
          scheme: 'bearer',
          bearerFormat: 'JWT',
          description: 'Enter JWT token',
        },
        'JWT-auth',
      )
      .addApiKey(
        {
          type: 'apiKey',
          name: 'X-API-Key',
          in: 'header',
          description: 'API Key for partner access',
        },
        'API-key',
      )
      // Core
      .addTag('auth', 'Authentication & Authorization')
      .addTag('users', 'User management')
      .addTag('drivers', 'Driver management')
      .addTag('bookings', 'Booking lifecycle')
      .addTag('payments', 'Payment processing')
      // Services
      .addTag('delivery', 'Package delivery')
      .addTag('handyman', 'On-demand services')
      .addTag('events', 'Events & venues')
      // Business
      .addTag('corporate', 'Corporate accounts')
      .addTag('partners', 'Partner integrations')
      .addTag('fleets', 'Fleet management')
      .addTag('franchises', 'Franchise management')
      // Operations
      .addTag('vehicles', 'Vehicle types & management')
      .addTag('zones', 'Geofencing & zones')
      .addTag('pricing', 'Pricing & surge')
      .addTag('promo', 'Promo codes & discounts')
      // Communications
      .addTag('notifications', 'Push, SMS, Email')
      .addTag('chat', 'In-app messaging')
      .addTag('websocket', 'Real-time events')
      // Support
      .addTag('support', 'Support tickets')
      .addTag('disputes', 'Disputes & claims')
      .addTag('ratings', 'Ratings & reviews')
      // Admin
      .addTag('admin', 'Admin operations')
      .addTag('reports', 'Reports & analytics')
      .addTag('settings', 'Dynamic configuration')
      .addTag('audit', 'Audit logs')
      // System
      .addTag('health', 'Health checks')
      .addTag('webhooks', 'Webhook management')
      .addTag('api-keys', 'API key management')
      .build();

    const document = SwaggerModule.createDocument(app, config);
    SwaggerModule.setup('docs', app, document, {
      swaggerOptions: {
        persistAuthorization: true,
        docExpansion: 'none',
        filter: true,
        showRequestDuration: true,
        syntaxHighlight: {
          activate: true,
          theme: 'monokai',
        },
      },
      customSiteTitle: 'MonkAPI Documentation',
      customfavIcon: '/favicon.ico',
      customCss: `
        .swagger-ui .topbar { display: none }
        .swagger-ui .info { margin: 20px 0 }
      `,
    });
  }

  const port = process.env.PORT || 3000;
  await app.listen(port);

  console.log(`
  ================================================
  MonkAPI Node.js Backend
  ================================================
  Environment: ${process.env.NODE_ENV || 'development'}
  Port: ${port}
  API Docs: http://localhost:${port}/docs
  ================================================
  `);
}

bootstrap();
