import {
  Injectable,
  CanActivate,
  ExecutionContext,
  BadRequestException,
} from '@nestjs/common';
import { Reflector } from '@nestjs/core';

/**
 * Merchant Guard - Verifie que merchant_id est present dans la requete
 *
 * Le projet Laravel est multi-tenant, chaque requete doit avoir un merchant_id
 */
@Injectable()
export class MerchantGuard implements CanActivate {
  constructor(private reflector: Reflector) {}

  canActivate(context: ExecutionContext): boolean {
    const request = context.switchToHttp().getRequest();

    // merchant_id peut etre dans body, query, ou headers
    const merchantId =
      request.body?.merchant_id ||
      request.query?.merchant_id ||
      request.headers['x-merchant-id'];

    if (!merchantId) {
      throw new BadRequestException('merchant_id is required');
    }

    // Attacher merchant_id a la requete pour utilisation ulterieure
    request.merchantId = parseInt(merchantId, 10);

    return true;
  }
}
