import { Module } from '@nestjs/common';
import { ConfigModule } from '@nestjs/config';
import { ThrottlerModule } from '@nestjs/throttler';
import { APP_FILTER, APP_INTERCEPTOR } from '@nestjs/core';

// Config
import appConfig from './config/app.config';
import databaseConfig from './config/database.config';
import jwtConfig from './config/jwt.config';
import redisConfig from './config/redis.config';

// Common
import { ResponseInterceptor } from './common/interceptors/response.interceptor';
import { HttpExceptionFilter } from './common/filters/http-exception.filter';
import { PrismaModule } from './common/prisma/prisma.module';

// Modules
import { AuthModule } from './modules/auth/auth.module';
import { UserModule } from './modules/user/user.module';
import { DriverModule } from './modules/driver/driver.module';
import { BookingModule } from './modules/booking/booking.module';
import { PaymentModule } from './modules/payment/payment.module';
import { NotificationModule } from './modules/notification/notification.module';
import { WebsocketModule } from './modules/websocket/websocket.module';
import { QueueModule } from './modules/queue/queue.module';
import { PricingModule } from './modules/pricing/pricing.module';
import { PromoModule } from './modules/promo/promo.module';
import { DeliveryModule } from './modules/delivery/delivery.module';
import { ReferralModule } from './modules/referral/referral.module';
import { BusinessSegmentModule } from './modules/business-segment/business-segment.module';
import { CorporateModule } from './modules/corporate/corporate.module';
import { RatingModule } from './modules/rating/rating.module';
import { ChatModule } from './modules/chat/chat.module';
import { DocumentModule } from './modules/document/document.module';
import { EventsModule } from './modules/events/events.module';
import { HandymanModule } from './modules/handyman/handyman.module';
import { FranchiseModule } from './modules/franchise/franchise.module';
import { WalletModule } from './modules/wallet/wallet.module';
import { AdminModule } from './modules/admin/admin.module';
import { SmsModule } from './modules/sms/sms.module';
import { StorageModule } from './modules/storage/storage.module';
import { CronModule } from './modules/cron/cron.module';
import { AnalyticsModule } from './modules/analytics/analytics.module';
import { EmailModule } from './modules/email/email.module';
import { GeocodingModule } from './modules/geocoding/geocoding.module';
import { HealthModule } from './modules/health/health.module';
import { I18nModule } from './modules/i18n/i18n.module';
import { AuditModule } from './modules/audit/audit.module';
import { CacheModule } from './modules/cache/cache.module';
import { WebhookModule } from './modules/webhook/webhook.module';
import { ApiKeyModule } from './modules/api-key/api-key.module';
import { SettingsModule } from './modules/settings/settings.module';
import { VehicleModule } from './modules/vehicle/vehicle.module';
import { ZoneModule } from './modules/zone/zone.module';
import { SupportModule } from './modules/support/support.module';
import { ReportModule } from './modules/report/report.module';
import { FleetModule } from './modules/fleet/fleet.module';
import { DisputeModule } from './modules/dispute/dispute.module';
import { PartnerModule } from './modules/partner/partner.module';

@Module({
  imports: [
    // Configuration
    ConfigModule.forRoot({
      isGlobal: true,
      load: [appConfig, databaseConfig, jwtConfig, redisConfig],
      envFilePath: ['.env', '.env.local'],
    }),

    // Rate limiting
    ThrottlerModule.forRoot([
      {
        ttl: 60000, // 1 minute
        limit: 100, // 100 requests per minute
      },
    ]),

    // Database
    PrismaModule,

    // Feature modules
    AuthModule,
    UserModule,
    DriverModule,
    BookingModule,
    PaymentModule,
    NotificationModule,

    // Real-time & Background jobs
    WebsocketModule,
    QueueModule,

    // Pricing & Promos
    PricingModule,
    PromoModule,

    // Delivery
    DeliveryModule,

    // Referral & Rewards
    ReferralModule,

    // Business Services
    BusinessSegmentModule,
    CorporateModule,

    // User Experience
    RatingModule,
    ChatModule,

    // Driver Management
    DocumentModule,

    // Events & Venues
    EventsModule,

    // On-demand Services
    HandymanModule,

    // Franchise Management
    FranchiseModule,

    // Wallet & Payments
    WalletModule,

    // Admin & Reports
    AdminModule,

    // SMS & Communications
    SmsModule,

    // File Storage
    StorageModule,

    // Scheduled Tasks
    CronModule,

    // Analytics & Metrics
    AnalyticsModule,

    // Email & Notifications
    EmailModule,

    // Geocoding & Maps
    GeocodingModule,

    // Health & Monitoring
    HealthModule,

    // Internationalization
    I18nModule,

    // Audit & Logging
    AuditModule,

    // Cache
    CacheModule,

    // Webhooks
    WebhookModule,

    // API Key Authentication
    ApiKeyModule,

    // Dynamic Configuration
    SettingsModule,

    // Vehicle Management
    VehicleModule,

    // Geofencing
    ZoneModule,

    // Support Tickets
    SupportModule,

    // Reports & Analytics
    ReportModule,

    // Fleet Management
    FleetModule,

    // Disputes & Claims
    DisputeModule,

    // Partners Integration
    PartnerModule,
  ],
  providers: [
    // Global response interceptor (format API standard)
    {
      provide: APP_INTERCEPTOR,
      useClass: ResponseInterceptor,
    },
    // Global exception filter
    {
      provide: APP_FILTER,
      useClass: HttpExceptionFilter,
    },
  ],
})
export class AppModule {}
