// =============================================================================
// MonkAPI - Data Migration Script from Laravel
// =============================================================================
//
// This script migrates data from the Laravel MySQL database to the new schema.
// Run with: npx ts-node scripts/migrate-data.ts
//
// Prerequisites:
// 1. Set LARAVEL_DATABASE_URL in .env (source DB)
// 2. Set DATABASE_URL in .env (target DB)
// 3. Ensure Prisma schema is generated
// =============================================================================

import { PrismaClient as TargetPrisma } from '@prisma/client';
import * as mysql from 'mysql2/promise';

// Configuration
const BATCH_SIZE = 1000;
const LARAVEL_DB_URL = process.env.LARAVEL_DATABASE_URL || '';

// Prisma client for target database
const prisma = new TargetPrisma();

// Connection pool for Laravel database
let laravelPool: mysql.Pool;

// Statistics
const stats = {
  merchants: 0,
  users: 0,
  drivers: 0,
  bookings: 0,
  vehicleTypes: 0,
  zones: 0,
  promoCodes: 0,
  errors: [] as string[],
};

// =============================================================================
// HELPER FUNCTIONS
// =============================================================================

function log(message: string, type: 'info' | 'success' | 'error' | 'warning' = 'info') {
  const icons = {
    info: '\x1b[36m[INFO]\x1b[0m',
    success: '\x1b[32m[SUCCESS]\x1b[0m',
    error: '\x1b[31m[ERROR]\x1b[0m',
    warning: '\x1b[33m[WARNING]\x1b[0m',
  };
  console.log(`${icons[type]} ${message}`);
}

function progress(current: number, total: number, label: string) {
  const pct = Math.round((current / total) * 100);
  const bar = '█'.repeat(Math.floor(pct / 5)) + '░'.repeat(20 - Math.floor(pct / 5));
  process.stdout.write(`\r${label}: [${bar}] ${pct}% (${current}/${total})`);
  if (current === total) console.log('');
}

async function queryLaravel<T>(sql: string, params: any[] = []): Promise<T[]> {
  const [rows] = await laravelPool.execute(sql, params);
  return rows as T[];
}

function transformDate(date: any): Date | null {
  if (!date) return null;
  const d = new Date(date);
  return isNaN(d.getTime()) ? null : d;
}

function transformDecimal(value: any): number | null {
  if (value === null || value === undefined) return null;
  const num = parseFloat(value);
  return isNaN(num) ? null : num;
}

// =============================================================================
// MIGRATION FUNCTIONS
// =============================================================================

async function migrateMerchants() {
  log('Migrating merchants...', 'info');

  const merchants = await queryLaravel<any>(`
    SELECT * FROM merchants WHERE deleted_at IS NULL
  `);

  for (const m of merchants) {
    try {
      await prisma.merchant.upsert({
        where: { id: m.id },
        update: {},
        create: {
          id: m.id,
          merchant_name: m.merchant_name,
          merchant_email: m.merchant_email,
          merchant_phone: m.merchant_phone,
          merchant_logo: m.merchant_logo,
          currency: m.currency || 'XOF',
          timezone: m.timezone || 'Africa/Lome',
          status: m.status || 1,
          created_at: transformDate(m.created_at),
          updated_at: transformDate(m.updated_at),
        },
      });
      stats.merchants++;
    } catch (error) {
      stats.errors.push(`Merchant ${m.id}: ${error}`);
    }
  }

  log(`Migrated ${stats.merchants} merchants`, 'success');
}

async function migrateUsers() {
  log('Migrating users...', 'info');

  const [countResult] = await queryLaravel<{ total: number }>(`
    SELECT COUNT(*) as total FROM users WHERE user_delete IS NULL
  `);
  const total = countResult.total;

  let offset = 0;
  while (offset < total) {
    const users = await queryLaravel<any>(`
      SELECT * FROM users
      WHERE user_delete IS NULL
      LIMIT ${BATCH_SIZE} OFFSET ${offset}
    `);

    for (const u of users) {
      try {
        await prisma.user.upsert({
          where: { id: u.id },
          update: {},
          create: {
            id: u.id,
            merchant_id: u.merchant_id,
            first_name: u.first_name,
            last_name: u.last_name,
            email: u.email,
            password: u.password,
            phone: u.UserPhone || u.phone,
            country_code: u.country_code,
            profile_image: u.profile_image,
            status: u.UserStatus || u.status || 1,
            device_type: u.device_type,
            player_id: u.player_id,
            fcm_token: u.fcm_token,
            language_id: u.language_id || 38,
            referral_code: u.referral_code,
            wallet_balance: transformDecimal(u.wallet_balance) || 0,
            rating: transformDecimal(u.rating) || 5.0,
            total_rides: u.total_rides || 0,
            created_at: transformDate(u.created_at),
            updated_at: transformDate(u.updated_at),
          },
        });
        stats.users++;
      } catch (error) {
        stats.errors.push(`User ${u.id}: ${error}`);
      }
    }

    offset += BATCH_SIZE;
    progress(Math.min(offset, total), total, 'Users');
  }

  log(`Migrated ${stats.users} users`, 'success');
}

async function migrateDrivers() {
  log('Migrating drivers...', 'info');

  const [countResult] = await queryLaravel<{ total: number }>(`
    SELECT COUNT(*) as total FROM drivers WHERE deleted_at IS NULL
  `);
  const total = countResult.total;

  let offset = 0;
  while (offset < total) {
    const drivers = await queryLaravel<any>(`
      SELECT * FROM drivers
      WHERE deleted_at IS NULL
      LIMIT ${BATCH_SIZE} OFFSET ${offset}
    `);

    for (const d of drivers) {
      try {
        await prisma.driver.upsert({
          where: { id: d.id },
          update: {},
          create: {
            id: d.id,
            merchant_id: d.merchant_id,
            first_name: d.first_name,
            last_name: d.last_name,
            email: d.email,
            password: d.password,
            phone: d.phoneNumber || d.phone,
            country_code: d.country_code,
            profile_image: d.profile_image,
            driver_status: d.driver_status || 0,
            is_online: d.is_online || 2,
            free_busy: d.free_busy || 2,
            latitude: transformDecimal(d.latitude),
            longitude: transformDecimal(d.longitude),
            device_type: d.device_type,
            player_id: d.player_id,
            fcm_token: d.fcm_token,
            language_id: d.language_id || 38,
            referral_code: d.referral_code,
            wallet_balance: transformDecimal(d.wallet_balance) || 0,
            rating: transformDecimal(d.rating) || 5.0,
            total_rides: d.total_rides || 0,
            total_earnings: transformDecimal(d.total_earnings) || 0,
            commission_rate: transformDecimal(d.commission_rate) || 20,
            created_at: transformDate(d.created_at),
            updated_at: transformDate(d.updated_at),
          },
        });
        stats.drivers++;
      } catch (error) {
        stats.errors.push(`Driver ${d.id}: ${error}`);
      }
    }

    offset += BATCH_SIZE;
    progress(Math.min(offset, total), total, 'Drivers');
  }

  log(`Migrated ${stats.drivers} drivers`, 'success');
}

async function migrateVehicleTypes() {
  log('Migrating vehicle types...', 'info');

  const types = await queryLaravel<any>(`
    SELECT * FROM vehicle_types WHERE deleted_at IS NULL
  `);

  for (const vt of types) {
    try {
      await prisma.vehicleType.upsert({
        where: { id: vt.id },
        update: {},
        create: {
          id: vt.id,
          merchant_id: vt.merchant_id,
          name: vt.VehicleName || vt.name,
          description: vt.description,
          icon: vt.icon || vt.VehicleImage,
          image: vt.image || vt.VehicleImage,
          capacity: vt.capacity || 4,
          base_fare: transformDecimal(vt.base_fare) || 0,
          per_km_fare: transformDecimal(vt.per_km_fare) || transformDecimal(vt.DistancePrice) || 0,
          per_minute_fare: transformDecimal(vt.per_minute_fare) || transformDecimal(vt.TimePrice) || 0,
          minimum_fare: transformDecimal(vt.minimum_fare) || transformDecimal(vt.BasePrice) || 0,
          cancellation_fee: transformDecimal(vt.cancellation_fee) || 0,
          waiting_fee: transformDecimal(vt.waiting_fee) || 0,
          is_active: vt.is_active !== 0 && vt.status !== 0,
          sort_order: vt.sort_order || 0,
          created_at: transformDate(vt.created_at),
          updated_at: transformDate(vt.updated_at),
        },
      });
      stats.vehicleTypes++;
    } catch (error) {
      stats.errors.push(`VehicleType ${vt.id}: ${error}`);
    }
  }

  log(`Migrated ${stats.vehicleTypes} vehicle types`, 'success');
}

async function migrateBookings() {
  log('Migrating bookings...', 'info');

  const [countResult] = await queryLaravel<{ total: number }>(`
    SELECT COUNT(*) as total FROM bookings
  `);
  const total = countResult.total;

  let offset = 0;
  while (offset < total) {
    const bookings = await queryLaravel<any>(`
      SELECT * FROM bookings
      LIMIT ${BATCH_SIZE} OFFSET ${offset}
    `);

    for (const b of bookings) {
      try {
        // Map Laravel booking status to new status
        const statusMap: Record<number, string> = {
          0: 'pending',
          1: 'accepted',
          2: 'arrived',
          3: 'started',
          4: 'completed',
          5: 'cancelled',
        };

        await prisma.booking.upsert({
          where: { id: b.id },
          update: {},
          create: {
            id: b.id,
            merchant_id: b.merchant_id,
            user_id: b.user_id,
            driver_id: b.driver_id,
            vehicle_type_id: b.VehicleTypeId || b.vehicle_type_id,
            booking_number: b.merchant_booking_id || b.booking_number,
            booking_type: b.booking_type || 'ride',
            booking_status: statusMap[b.BookingStatus] || b.booking_status || 'pending',
            payment_status: b.payment_status || 0,
            payment_method: b.PaymentType === 1 ? 'cash' : b.PaymentType === 2 ? 'card' : 'wallet',
            pickup_latitude: transformDecimal(b.PickupLat || b.pickup_latitude),
            pickup_longitude: transformDecimal(b.PickupLng || b.pickup_longitude),
            pickup_address: b.PickupAddress || b.pickup_address,
            drop_latitude: transformDecimal(b.DropLat || b.drop_latitude),
            drop_longitude: transformDecimal(b.DropLng || b.drop_longitude),
            drop_address: b.DropAddress || b.drop_address,
            estimate_amount: transformDecimal(b.estimate_amount) || transformDecimal(b.BookingTotalAmount),
            final_amount: transformDecimal(b.final_amount) || transformDecimal(b.TotalAmount),
            travel_distance: transformDecimal(b.travel_distance) || transformDecimal(b.TotalDistance),
            travel_time: b.travel_time || b.TotalTime,
            surge_multiplier: transformDecimal(b.surge_multiplier) || 1,
            discount_amount: transformDecimal(b.discount_amount) || 0,
            booking_time: transformDate(b.booking_time) || transformDate(b.created_at),
            accepted_time: transformDate(b.accepted_time) || transformDate(b.AcceptedTime),
            arrived_time: transformDate(b.arrived_time) || transformDate(b.ArrivedTime),
            started_time: transformDate(b.started_time) || transformDate(b.StartedTime),
            completed_time: transformDate(b.completed_time) || transformDate(b.CompletedTime),
            cancelled_time: transformDate(b.cancelled_time),
            created_at: transformDate(b.created_at),
            updated_at: transformDate(b.updated_at),
          },
        });
        stats.bookings++;
      } catch (error) {
        stats.errors.push(`Booking ${b.id}: ${error}`);
      }
    }

    offset += BATCH_SIZE;
    progress(Math.min(offset, total), total, 'Bookings');
  }

  log(`Migrated ${stats.bookings} bookings`, 'success');
}

async function migrateZones() {
  log('Migrating zones...', 'info');

  // Try different table names that might exist in Laravel
  const tableNames = ['zones', 'geofences', 'service_areas'];
  let zones: any[] = [];

  for (const table of tableNames) {
    try {
      zones = await queryLaravel<any>(`SELECT * FROM ${table} WHERE deleted_at IS NULL`);
      if (zones.length > 0) break;
    } catch {
      // Table doesn't exist, try next
    }
  }

  for (const z of zones) {
    try {
      await prisma.zone.upsert({
        where: { id: z.id },
        update: {},
        create: {
          id: z.id,
          merchant_id: z.merchant_id,
          name: z.name,
          description: z.description,
          type: z.type || 'service_area',
          shape: z.shape || 'polygon',
          coordinates: z.coordinates ? JSON.parse(z.coordinates) : null,
          center_lat: transformDecimal(z.center_lat),
          center_lng: transformDecimal(z.center_lng),
          radius: transformDecimal(z.radius),
          surge_multiplier: transformDecimal(z.surge_multiplier) || 1,
          extra_fee: transformDecimal(z.extra_fee) || 0,
          is_active: z.is_active !== 0,
          created_at: transformDate(z.created_at),
          updated_at: transformDate(z.updated_at),
        },
      });
      stats.zones++;
    } catch (error) {
      stats.errors.push(`Zone ${z.id}: ${error}`);
    }
  }

  log(`Migrated ${stats.zones} zones`, 'success');
}

async function migratePromoCodes() {
  log('Migrating promo codes...', 'info');

  const promos = await queryLaravel<any>(`
    SELECT * FROM promo_codes WHERE deleted_at IS NULL
  `);

  for (const p of promos) {
    try {
      await prisma.promoCode.upsert({
        where: { id: p.id },
        update: {},
        create: {
          id: p.id,
          merchant_id: p.merchant_id,
          code: p.code || p.PromoCode,
          name: p.name || p.PromoName,
          description: p.description,
          type: p.type === 1 ? 'percentage' : 'fixed',
          discount_value: transformDecimal(p.discount_value) || transformDecimal(p.PromoValue),
          max_discount: transformDecimal(p.max_discount),
          min_order_value: transformDecimal(p.min_order_value),
          usage_limit: p.usage_limit || p.UsageLimit,
          usage_per_user: p.usage_per_user || 1,
          used_count: p.used_count || 0,
          valid_from: transformDate(p.valid_from) || transformDate(p.StartDate),
          valid_until: transformDate(p.valid_until) || transformDate(p.EndDate),
          is_active: p.is_active !== 0 && p.status !== 0,
          created_at: transformDate(p.created_at),
          updated_at: transformDate(p.updated_at),
        },
      });
      stats.promoCodes++;
    } catch (error) {
      stats.errors.push(`PromoCode ${p.id}: ${error}`);
    }
  }

  log(`Migrated ${stats.promoCodes} promo codes`, 'success');
}

// =============================================================================
// MAIN MIGRATION
// =============================================================================

async function main() {
  console.log('\n');
  console.log('═'.repeat(60));
  console.log('MonkAPI - Laravel to Node.js Data Migration');
  console.log('═'.repeat(60));
  console.log('\n');

  // Parse Laravel database URL
  if (!LARAVEL_DB_URL) {
    log('LARAVEL_DATABASE_URL not set in environment', 'error');
    process.exit(1);
  }

  // Create Laravel connection pool
  const url = new URL(LARAVEL_DB_URL);
  laravelPool = mysql.createPool({
    host: url.hostname,
    port: parseInt(url.port) || 3306,
    user: url.username,
    password: url.password,
    database: url.pathname.slice(1),
    waitForConnections: true,
    connectionLimit: 10,
  });

  log('Connected to Laravel database', 'success');

  try {
    // Run migrations in order
    const startTime = Date.now();

    await migrateMerchants();
    await migrateVehicleTypes();
    await migrateUsers();
    await migrateDrivers();
    await migrateBookings();
    await migrateZones();
    await migratePromoCodes();

    const duration = Math.round((Date.now() - startTime) / 1000);

    // Print summary
    console.log('\n');
    console.log('═'.repeat(60));
    console.log('Migration Summary');
    console.log('═'.repeat(60));
    console.log(`  Merchants:     ${stats.merchants}`);
    console.log(`  Users:         ${stats.users}`);
    console.log(`  Drivers:       ${stats.drivers}`);
    console.log(`  Bookings:      ${stats.bookings}`);
    console.log(`  Vehicle Types: ${stats.vehicleTypes}`);
    console.log(`  Zones:         ${stats.zones}`);
    console.log(`  Promo Codes:   ${stats.promoCodes}`);
    console.log(`  Duration:      ${duration}s`);
    console.log(`  Errors:        ${stats.errors.length}`);
    console.log('═'.repeat(60));

    if (stats.errors.length > 0) {
      console.log('\nErrors:');
      stats.errors.slice(0, 10).forEach((e) => console.log(`  - ${e}`));
      if (stats.errors.length > 10) {
        console.log(`  ... and ${stats.errors.length - 10} more`);
      }
    }

    console.log('\n');
    log('Migration completed!', 'success');
  } catch (error) {
    log(`Migration failed: ${error}`, 'error');
    process.exit(1);
  } finally {
    await laravelPool.end();
    await prisma.$disconnect();
  }
}

main();
