// =============================================================================
// MonkAPI - Database Seed
// Run with: npx prisma db seed
// =============================================================================

import { PrismaClient } from '@prisma/client';
import * as bcrypt from 'bcrypt';

const prisma = new PrismaClient();

async function main() {
  console.log('Starting database seed...');

  // ==========================================================================
  // LANGUAGES
  // ==========================================================================
  console.log('Seeding languages...');

  const languages = [
    { code: 'en', name: 'English', native_name: 'English', rtl: false },
    { code: 'fr', name: 'French', native_name: 'Francais', rtl: false },
    { code: 'ar', name: 'Arabic', native_name: 'العربية', rtl: true },
    { code: 'pt', name: 'Portuguese', native_name: 'Portugues', rtl: false },
    { code: 'es', name: 'Spanish', native_name: 'Espanol', rtl: false },
  ];

  for (const lang of languages) {
    await prisma.language.upsert({
      where: { code: lang.code },
      update: {},
      create: lang,
    });
  }

  // ==========================================================================
  // COUNTRIES
  // ==========================================================================
  console.log('Seeding countries...');

  const countries = [
    { name: 'Togo', code: 'TG', phone_code: '+228', currency_code: 'XOF' },
    { name: 'Benin', code: 'BJ', phone_code: '+229', currency_code: 'XOF' },
    { name: 'Ghana', code: 'GH', phone_code: '+233', currency_code: 'GHS' },
    { name: 'Ivory Coast', code: 'CI', phone_code: '+225', currency_code: 'XOF' },
    { name: 'Senegal', code: 'SN', phone_code: '+221', currency_code: 'XOF' },
    { name: 'Nigeria', code: 'NG', phone_code: '+234', currency_code: 'NGN' },
    { name: 'Cameroon', code: 'CM', phone_code: '+237', currency_code: 'XAF' },
    { name: 'Kenya', code: 'KE', phone_code: '+254', currency_code: 'KES' },
    { name: 'South Africa', code: 'ZA', phone_code: '+27', currency_code: 'ZAR' },
    { name: 'Morocco', code: 'MA', phone_code: '+212', currency_code: 'MAD' },
    { name: 'Egypt', code: 'EG', phone_code: '+20', currency_code: 'EGP' },
    { name: 'France', code: 'FR', phone_code: '+33', currency_code: 'EUR' },
    { name: 'United States', code: 'US', phone_code: '+1', currency_code: 'USD' },
  ];

  for (const country of countries) {
    await prisma.country.upsert({
      where: { code: country.code },
      update: {},
      create: country,
    });
  }

  // ==========================================================================
  // CURRENCIES
  // ==========================================================================
  console.log('Seeding currencies...');

  const currencies = [
    { code: 'XOF', name: 'CFA Franc BCEAO', symbol: 'FCFA', decimal_places: 0, exchange_rate: 655.957 },
    { code: 'XAF', name: 'CFA Franc BEAC', symbol: 'FCFA', decimal_places: 0, exchange_rate: 655.957 },
    { code: 'EUR', name: 'Euro', symbol: '€', decimal_places: 2, exchange_rate: 1 },
    { code: 'USD', name: 'US Dollar', symbol: '$', decimal_places: 2, exchange_rate: 1.08 },
    { code: 'GHS', name: 'Ghanaian Cedi', symbol: 'GH₵', decimal_places: 2, exchange_rate: 12.5 },
    { code: 'NGN', name: 'Nigerian Naira', symbol: '₦', decimal_places: 2, exchange_rate: 1650 },
    { code: 'KES', name: 'Kenyan Shilling', symbol: 'KSh', decimal_places: 2, exchange_rate: 156 },
    { code: 'ZAR', name: 'South African Rand', symbol: 'R', decimal_places: 2, exchange_rate: 19.5 },
    { code: 'MAD', name: 'Moroccan Dirham', symbol: 'DH', decimal_places: 2, exchange_rate: 10.8 },
    { code: 'EGP', name: 'Egyptian Pound', symbol: 'E£', decimal_places: 2, exchange_rate: 52 },
    { code: 'GBP', name: 'British Pound', symbol: '£', decimal_places: 2, exchange_rate: 0.86 },
  ];

  for (const currency of currencies) {
    await prisma.currency.upsert({
      where: { code: currency.code },
      update: {},
      create: currency,
    });
  }

  // ==========================================================================
  // DEMO MERCHANT
  // ==========================================================================
  console.log('Seeding demo merchant...');

  const merchant = await prisma.merchant.upsert({
    where: { id: 1 },
    update: {},
    create: {
      id: 1,
      merchant_name: 'MonkRide Demo',
      merchant_email: 'demo@monkapi.com',
      merchant_phone: '+22890000000',
      currency: 'XOF',
      timezone: 'Africa/Lome',
      status: 1,
    },
  });

  // ==========================================================================
  // VEHICLE TYPES
  // ==========================================================================
  console.log('Seeding vehicle types...');

  const vehicleTypes = [
    {
      name: 'Moto',
      description: 'Moto-taxi rapide pour trajets courts',
      capacity: 1,
      base_fare: 300,
      per_km_fare: 100,
      per_minute_fare: 15,
      minimum_fare: 500,
      cancellation_fee: 200,
      waiting_fee: 10,
      sort_order: 1,
    },
    {
      name: 'Economique',
      description: 'Voiture economique pour trajets quotidiens',
      capacity: 4,
      base_fare: 500,
      per_km_fare: 200,
      per_minute_fare: 25,
      minimum_fare: 1000,
      cancellation_fee: 500,
      waiting_fee: 20,
      sort_order: 2,
    },
    {
      name: 'Confort',
      description: 'Voiture confortable avec climatisation',
      capacity: 4,
      base_fare: 800,
      per_km_fare: 300,
      per_minute_fare: 35,
      minimum_fare: 1500,
      cancellation_fee: 750,
      waiting_fee: 30,
      sort_order: 3,
    },
    {
      name: 'Premium',
      description: 'Vehicule premium pour VIP',
      capacity: 4,
      base_fare: 1500,
      per_km_fare: 500,
      per_minute_fare: 50,
      minimum_fare: 3000,
      cancellation_fee: 1500,
      waiting_fee: 50,
      sort_order: 4,
    },
    {
      name: 'Van',
      description: 'Van pour groupes ou bagages',
      capacity: 7,
      base_fare: 1000,
      per_km_fare: 400,
      per_minute_fare: 40,
      minimum_fare: 2000,
      cancellation_fee: 1000,
      waiting_fee: 40,
      sort_order: 5,
    },
    {
      name: 'Livraison',
      description: 'Service de livraison de colis',
      capacity: 0,
      base_fare: 500,
      per_km_fare: 150,
      per_minute_fare: 20,
      minimum_fare: 1000,
      cancellation_fee: 500,
      waiting_fee: 15,
      sort_order: 6,
    },
  ];

  for (const vt of vehicleTypes) {
    await prisma.vehicleType.upsert({
      where: { id: vehicleTypes.indexOf(vt) + 1 },
      update: {},
      create: {
        ...vt,
        merchant_id: merchant.id,
      },
    });
  }

  // ==========================================================================
  // BUSINESS SEGMENTS
  // ==========================================================================
  console.log('Seeding business segments...');

  const segments = [
    { name: 'Ride', slug: 'ride', type: 'ride', color: '#4CAF50', description: 'Transport de personnes' },
    { name: 'Delivery', slug: 'delivery', type: 'delivery', color: '#FF9800', description: 'Livraison de colis' },
    { name: 'Rental', slug: 'rental', type: 'rental', color: '#2196F3', description: 'Location de vehicules' },
    { name: 'Handyman', slug: 'handyman', type: 'handyman', color: '#9C27B0', description: 'Services a domicile' },
  ];

  for (const segment of segments) {
    await prisma.businessSegment.upsert({
      where: { merchant_id_slug: { merchant_id: merchant.id, slug: segment.slug } },
      update: {},
      create: {
        ...segment,
        merchant_id: merchant.id,
      },
    });
  }

  // ==========================================================================
  // DEFAULT SETTINGS
  // ==========================================================================
  console.log('Seeding default settings...');

  const settings = [
    // App Settings
    { category: 'app', key: 'app_name', value: 'MonkRide', type: 'string', is_public: true },
    { category: 'app', key: 'app_logo', value: '/assets/logo.png', type: 'string', is_public: true },
    { category: 'app', key: 'support_email', value: 'support@monkride.com', type: 'string', is_public: true },
    { category: 'app', key: 'support_phone', value: '+22890000000', type: 'string', is_public: true },

    // Booking Settings
    { category: 'booking', key: 'driver_search_radius', value: '5', type: 'number', is_public: false },
    { category: 'booking', key: 'max_search_attempts', value: '5', type: 'number', is_public: false },
    { category: 'booking', key: 'driver_response_timeout', value: '30', type: 'number', is_public: false },
    { category: 'booking', key: 'allow_scheduled_bookings', value: 'true', type: 'boolean', is_public: true },
    { category: 'booking', key: 'max_advance_booking_days', value: '7', type: 'number', is_public: true },

    // Payment Settings
    { category: 'payment', key: 'allow_cash', value: 'true', type: 'boolean', is_public: true },
    { category: 'payment', key: 'allow_card', value: 'true', type: 'boolean', is_public: true },
    { category: 'payment', key: 'allow_wallet', value: 'true', type: 'boolean', is_public: true },
    { category: 'payment', key: 'allow_mobile_money', value: 'true', type: 'boolean', is_public: true },
    { category: 'payment', key: 'min_wallet_topup', value: '1000', type: 'number', is_public: true },
    { category: 'payment', key: 'max_wallet_topup', value: '500000', type: 'number', is_public: true },

    // Surge Settings
    { category: 'surge', key: 'enable_surge', value: 'true', type: 'boolean', is_public: false },
    { category: 'surge', key: 'surge_threshold', value: '0.8', type: 'number', is_public: false },
    { category: 'surge', key: 'max_surge_multiplier', value: '3.0', type: 'number', is_public: false },

    // Driver Settings
    { category: 'driver', key: 'default_commission_rate', value: '20', type: 'number', is_public: false },
    { category: 'driver', key: 'min_payout_amount', value: '5000', type: 'number', is_public: false },
    { category: 'driver', key: 'auto_approve_drivers', value: 'false', type: 'boolean', is_public: false },

    // Notification Settings
    { category: 'notification', key: 'enable_push', value: 'true', type: 'boolean', is_public: false },
    { category: 'notification', key: 'enable_sms', value: 'true', type: 'boolean', is_public: false },
    { category: 'notification', key: 'enable_email', value: 'true', type: 'boolean', is_public: false },
  ];

  for (const setting of settings) {
    await prisma.setting.upsert({
      where: { merchant_id_key: { merchant_id: merchant.id, key: setting.key } },
      update: {},
      create: {
        ...setting,
        merchant_id: merchant.id,
      },
    });
  }

  // ==========================================================================
  // DEMO ADMIN USER
  // ==========================================================================
  console.log('Seeding demo admin user...');

  const hashedPassword = await bcrypt.hash('admin123', 10);

  await prisma.adminUser.upsert({
    where: { email: 'admin@monkapi.com' },
    update: {},
    create: {
      merchant_id: merchant.id,
      first_name: 'Admin',
      last_name: 'Demo',
      email: 'admin@monkapi.com',
      password: hashedPassword,
      phone: '+22890000001',
      role: 'super_admin',
      is_active: true,
    },
  });

  // ==========================================================================
  // DEMO ZONES
  // ==========================================================================
  console.log('Seeding demo zones...');

  const zones = [
    {
      name: 'Lome Centre',
      type: 'service_area',
      shape: 'polygon',
      coordinates: [
        [6.1256, 1.2077],
        [6.1256, 1.2477],
        [6.1656, 1.2477],
        [6.1656, 1.2077],
        [6.1256, 1.2077],
      ],
      color: '#4CAF50',
      is_active: true,
    },
    {
      name: 'Aeroport de Lome',
      type: 'airport',
      shape: 'circle',
      center_lat: 6.1654,
      center_lng: 1.2544,
      radius: 2000,
      extra_fee: 500,
      color: '#2196F3',
      is_active: true,
    },
    {
      name: 'Zone Surge - Marche Assigame',
      type: 'surge_zone',
      shape: 'circle',
      center_lat: 6.1319,
      center_lng: 1.2228,
      radius: 500,
      surge_multiplier: 1.5,
      color: '#FF9800',
      is_active: true,
    },
  ];

  for (const zone of zones) {
    await prisma.zone.create({
      data: {
        ...zone,
        merchant_id: merchant.id,
        coordinates: zone.coordinates ? JSON.stringify(zone.coordinates) : null,
      },
    });
  }

  // ==========================================================================
  // HANDYMAN SERVICES
  // ==========================================================================
  console.log('Seeding handyman services...');

  const handymanServices = [
    { name: 'Plomberie', category: 'plumbing', base_price: 5000, price_type: 'hourly', duration_minutes: 60 },
    { name: 'Electricite', category: 'electrical', base_price: 6000, price_type: 'hourly', duration_minutes: 60 },
    { name: 'Menage', category: 'cleaning', base_price: 10000, price_type: 'fixed', duration_minutes: 180 },
    { name: 'Jardinage', category: 'gardening', base_price: 8000, price_type: 'fixed', duration_minutes: 120 },
    { name: 'Peinture', category: 'painting', base_price: 7000, price_type: 'hourly', duration_minutes: 60 },
    { name: 'Menuiserie', category: 'carpentry', base_price: 8000, price_type: 'hourly', duration_minutes: 60 },
    { name: 'Climatisation', category: 'hvac', base_price: 10000, price_type: 'hourly', duration_minutes: 60 },
    { name: 'Serrurerie', category: 'locksmith', base_price: 5000, price_type: 'fixed', duration_minutes: 30 },
  ];

  for (const service of handymanServices) {
    await prisma.handymanService.create({
      data: {
        ...service,
        merchant_id: merchant.id,
      },
    });
  }

  // ==========================================================================
  // REFERRAL PROGRAM
  // ==========================================================================
  console.log('Seeding referral program...');

  await prisma.referralProgram.create({
    data: {
      merchant_id: merchant.id,
      name: 'Programme de Parrainage',
      referrer_reward_type: 'credit',
      referrer_reward_value: 1000,
      referee_reward_type: 'credit',
      referee_reward_value: 500,
      min_rides_to_unlock: 1,
      is_active: true,
    },
  });

  // ==========================================================================
  // DEMO PROMO CODES
  // ==========================================================================
  console.log('Seeding demo promo codes...');

  const promoCodes = [
    {
      code: 'WELCOME',
      name: 'Bienvenue',
      type: 'percentage',
      discount_value: 20,
      max_discount: 2000,
      usage_limit: 1000,
      usage_per_user: 1,
      new_users_only: true,
      first_ride_only: true,
    },
    {
      code: 'RIDE50',
      name: '50% de reduction',
      type: 'percentage',
      discount_value: 50,
      max_discount: 5000,
      usage_limit: 100,
      usage_per_user: 1,
      valid_until: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days
    },
    {
      code: 'FREE1000',
      name: '1000 FCFA offerts',
      type: 'fixed',
      discount_value: 1000,
      usage_limit: 500,
      usage_per_user: 2,
    },
  ];

  for (const promo of promoCodes) {
    await prisma.promoCode.upsert({
      where: { merchant_id_code: { merchant_id: merchant.id, code: promo.code } },
      update: {},
      create: {
        ...promo,
        merchant_id: merchant.id,
      },
    });
  }

  console.log('');
  console.log('='.repeat(60));
  console.log('Database seeded successfully!');
  console.log('='.repeat(60));
  console.log('');
  console.log('Demo credentials:');
  console.log('  Admin: admin@monkapi.com / admin123');
  console.log('');
  console.log('Demo merchant ID: 1');
  console.log('');
}

main()
  .catch((e) => {
    console.error('Seed error:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
