# Guide de Contribution

Merci de contribuer a MonkAPI! Ce guide vous aidera a demarrer.

## Code de conduite

- Soyez respectueux et inclusif
- Acceptez les critiques constructives
- Concentrez-vous sur ce qui est meilleur pour le projet

---

## Demarrage rapide

### 1. Fork et clone

```bash
# Fork sur GitHub puis clone
git clone https://github.com/VOTRE_USERNAME/monkapi-node.git
cd monkapi-node

# Ajouter l'upstream
git remote add upstream https://github.com/monkapi/api.git
```

### 2. Installation

```bash
# Installer les dependances
pnpm install

# Copier la configuration
cp .env.example .env

# Demarrer les services
docker-compose up -d mysql redis

# Generer Prisma
pnpm prisma generate
pnpm prisma migrate dev

# Demarrer le serveur
pnpm start:dev
```

### 3. Creer une branche

```bash
# Synchroniser avec upstream
git fetch upstream
git checkout main
git merge upstream/main

# Creer une branche
git checkout -b feature/ma-nouvelle-fonctionnalite
```

---

## Standards de code

### TypeScript

- Utiliser TypeScript strict
- Pas de `any` sauf cas exceptionnels
- Documenter les fonctions publiques

```typescript
// Bon
export async function calculateFare(
  distance: number,
  duration: number,
  vehicleType: VehicleType,
): Promise<FareEstimate> {
  // ...
}

// Mauvais
export async function calculateFare(distance, duration, vehicleType) {
  // ...
}
```

### Nommage

| Type | Convention | Exemple |
|------|------------|---------|
| Classes | PascalCase | `BookingService` |
| Interfaces | PascalCase avec I | `IBookingService` |
| Fonctions | camelCase | `calculateFare` |
| Variables | camelCase | `totalAmount` |
| Constantes | UPPER_SNAKE | `MAX_RETRY_COUNT` |
| Fichiers | kebab-case | `booking.service.ts` |
| Dossiers | kebab-case | `payment-methods/` |

### Structure des fichiers

```typescript
// 1. Imports externes
import { Injectable } from '@nestjs/common';
import { PrismaService } from '@/common/prisma/prisma.service';

// 2. Imports internes
import { CreateBookingDto } from './dto/create-booking.dto';

// 3. Constants/Types
const MAX_DISTANCE = 100;

interface BookingResult {
  // ...
}

// 4. Decorator
@Injectable()
export class BookingService {
  // 5. Constructor
  constructor(private readonly prisma: PrismaService) {}

  // 6. Public methods
  async create(dto: CreateBookingDto): Promise<Booking> {
    // ...
  }

  // 7. Private methods
  private calculateFare(): number {
    // ...
  }
}
```

### Formatting

```bash
# Formater le code
pnpm format

# Verifier le linting
pnpm lint

# Verifier les types
pnpm type-check
```

Configuration ESLint et Prettier incluse dans le projet.

---

## Commits

### Format des messages

```
<type>(<scope>): <description>

[body optionnel]

[footer optionnel]
```

### Types

| Type | Description |
|------|-------------|
| `feat` | Nouvelle fonctionnalite |
| `fix` | Correction de bug |
| `docs` | Documentation |
| `style` | Formatage (pas de changement de code) |
| `refactor` | Refactoring |
| `perf` | Amelioration performance |
| `test` | Ajout/modification de tests |
| `build` | Build, dependencies |
| `ci` | CI/CD |
| `chore` | Maintenance |

### Exemples

```bash
# Bon
git commit -m "feat(booking): add surge pricing calculation"
git commit -m "fix(auth): handle expired OTP correctly"
git commit -m "docs(api): update booking endpoint documentation"

# Mauvais
git commit -m "fixed stuff"
git commit -m "WIP"
git commit -m "update"
```

---

## Pull Requests

### Checklist

- [ ] Le code compile sans erreurs
- [ ] Les tests passent (`pnpm test`)
- [ ] Le linting passe (`pnpm lint`)
- [ ] La documentation est mise a jour si necessaire
- [ ] Le titre de la PR suit le format conventionnel

### Template PR

```markdown
## Description

[Decrivez les changements]

## Type de changement

- [ ] Bug fix
- [ ] Nouvelle fonctionnalite
- [ ] Breaking change
- [ ] Documentation

## Comment tester

1. [Etape 1]
2. [Etape 2]

## Checklist

- [ ] Tests ajoutes
- [ ] Documentation mise a jour
- [ ] Pas de breaking changes (ou documente)
```

### Process de review

1. Creer la PR vers `main`
2. Attendre les checks CI
3. Demander une review
4. Adresser les commentaires
5. Merge apres approbation

---

## Tests

### Structure

```
test/
├── setup.ts              # Configuration globale
├── factories/            # Factories pour tests
├── unit/                 # Tests unitaires
│   └── *.spec.ts
└── e2e/                  # Tests E2E
    └── *.e2e-spec.ts
```

### Tests unitaires

```typescript
// booking.service.spec.ts
describe('BookingService', () => {
  let service: BookingService;
  let prisma: PrismaService;

  beforeEach(async () => {
    const module = await Test.createTestingModule({
      providers: [
        BookingService,
        { provide: PrismaService, useValue: mockPrismaService },
      ],
    }).compile();

    service = module.get<BookingService>(BookingService);
  });

  describe('create', () => {
    it('should create a booking', async () => {
      const dto = createBookingDto();
      const result = await service.create(1, dto);
      expect(result.booking_status).toBe('pending');
    });

    it('should throw if zone not serviced', async () => {
      const dto = createBookingDto({ pickup_latitude: 0, pickup_longitude: 0 });
      await expect(service.create(1, dto)).rejects.toThrow(ZoneNotServicedException);
    });
  });
});
```

### Tests E2E

```typescript
// auth.e2e-spec.ts
describe('Auth (e2e)', () => {
  let app: INestApplication;

  beforeAll(async () => {
    const module = await Test.createTestingModule({
      imports: [AppModule],
    }).compile();

    app = module.createNestApplication();
    await app.init();
  });

  it('/auth/login (POST)', () => {
    return request(app.getHttpServer())
      .post('/api/v1/auth/login')
      .send({ email: 'test@example.com', password: 'password' })
      .expect(200)
      .expect((res) => {
        expect(res.body.data.access_token).toBeDefined();
      });
  });
});
```

### Commandes

```bash
# Tous les tests
pnpm test

# Tests avec coverage
pnpm test:cov

# Tests E2E
pnpm test:e2e

# Tests en watch mode
pnpm test:watch

# Test d'un fichier specifique
pnpm test -- booking.service.spec.ts
```

---

## Documentation

### Code

Documenter les fonctions publiques avec JSDoc:

```typescript
/**
 * Calcule le tarif estime pour une course.
 *
 * @param distance - Distance en kilometres
 * @param duration - Duree estimee en minutes
 * @param vehicleType - Type de vehicule
 * @returns Estimation du tarif
 * @throws {ZoneNotServicedException} Si la zone n'est pas desservie
 *
 * @example
 * const fare = await calculateFare(10, 25, vehicleType);
 * // { amount: 3500, currency: 'XOF' }
 */
async calculateFare(
  distance: number,
  duration: number,
  vehicleType: VehicleType,
): Promise<FareEstimate> {
  // ...
}
```

### API (Swagger)

Utiliser les decorateurs Swagger:

```typescript
@ApiOperation({ summary: 'Creer une reservation' })
@ApiResponse({ status: 201, description: 'Reservation creee', type: BookingResponseDto })
@ApiResponse({ status: 400, description: 'Donnees invalides' })
@ApiResponse({ status: 404, description: 'Zone non desservie' })
@Post()
async create(@Body() dto: CreateBookingDto): Promise<Booking> {
  // ...
}
```

### Fichiers docs/

Mettre a jour les fichiers dans `docs/` si vous modifiez:
- L'architecture
- Les endpoints API
- La configuration
- Le deploiement

---

## Structure des modules

### Creer un nouveau module

```bash
# Generer les fichiers
nest g module modules/my-feature
nest g controller modules/my-feature
nest g service modules/my-feature
```

### Structure recommandee

```
src/modules/my-feature/
├── my-feature.module.ts      # Declaration du module
├── my-feature.controller.ts  # Endpoints HTTP
├── my-feature.service.ts     # Logique metier
├── my-feature.gateway.ts     # WebSocket (optionnel)
├── dto/
│   ├── create-my-feature.dto.ts
│   └── update-my-feature.dto.ts
├── entities/
│   └── my-feature.entity.ts
├── guards/
│   └── my-feature.guard.ts
├── events/
│   └── my-feature.events.ts
└── __tests__/
    └── my-feature.service.spec.ts
```

### Enregistrement

```typescript
// my-feature.module.ts
@Module({
  imports: [PrismaModule],
  controllers: [MyFeatureController],
  providers: [MyFeatureService],
  exports: [MyFeatureService],
})
export class MyFeatureModule {}

// app.module.ts
@Module({
  imports: [
    // ...
    MyFeatureModule,
  ],
})
export class AppModule {}
```

---

## Questions?

- Ouvrir une issue sur GitHub
- Rejoindre le channel Slack #monkapi-dev
- Contacter l'equipe: dev@monkapi.com
