# Architecture MonkAPI

## Vue d'ensemble

MonkAPI utilise une architecture modulaire basee sur NestJS, permettant une separation claire des responsabilites et une scalabilite horizontale.

```
┌─────────────────────────────────────────────────────────────────┐
│                         Load Balancer                            │
│                      (NGINX Ingress)                             │
└─────────────────────────────────────────────────────────────────┘
                                │
                ┌───────────────┼───────────────┐
                │               │               │
                ▼               ▼               ▼
        ┌───────────┐   ┌───────────┐   ┌───────────┐
        │  Pod #1   │   │  Pod #2   │   │  Pod #3   │
        │  MonkAPI  │   │  MonkAPI  │   │  MonkAPI  │
        └─────┬─────┘   └─────┬─────┘   └─────┬─────┘
              │               │               │
              └───────────────┼───────────────┘
                              │
        ┌─────────────────────┼─────────────────────┐
        │                     │                     │
        ▼                     ▼                     ▼
┌───────────────┐     ┌───────────────┐     ┌───────────────┐
│    MySQL      │     │    Redis      │     │      S3       │
│   (Primary)   │     │  (Cache/MQ)   │     │  (Storage)    │
└───────────────┘     └───────────────┘     └───────────────┘
```

## Couches applicatives

```
┌─────────────────────────────────────────────────────────────────┐
│                      Presentation Layer                          │
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐              │
│  │ Controllers │  │   Guards    │  │Interceptors │              │
│  └─────────────┘  └─────────────┘  └─────────────┘              │
├─────────────────────────────────────────────────────────────────┤
│                       Business Layer                             │
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐              │
│  │  Services   │  │   Events    │  │    Jobs     │              │
│  └─────────────┘  └─────────────┘  └─────────────┘              │
├─────────────────────────────────────────────────────────────────┤
│                         Data Layer                               │
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐              │
│  │   Prisma    │  │    Redis    │  │  External   │              │
│  │    ORM      │  │    Cache    │  │    APIs     │              │
│  └─────────────┘  └─────────────┘  └─────────────┘              │
└─────────────────────────────────────────────────────────────────┘
```

## Architecture des modules

### Structure d'un module NestJS

```
src/modules/booking/
├── booking.module.ts       # Declaration du module
├── booking.controller.ts   # Endpoints HTTP
├── booking.service.ts      # Logique metier
├── booking.gateway.ts      # WebSocket (optionnel)
├── dto/                    # Data Transfer Objects
│   ├── create-booking.dto.ts
│   ├── update-booking.dto.ts
│   └── estimate.dto.ts
├── entities/               # Types/Interfaces
│   └── booking.entity.ts
├── guards/                 # Guards specifiques
│   └── booking-owner.guard.ts
├── events/                 # Event handlers
│   └── booking-created.event.ts
└── jobs/                   # Background jobs
    └── assign-driver.job.ts
```

### Flux d'une requete

```
Request
   │
   ▼
┌──────────────────┐
│   Middleware     │  ← Logging, CORS, Compression
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│     Guards       │  ← Authentication, Authorization
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│  Interceptors    │  ← Transform request/response
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│    Pipes         │  ← Validation, Transformation
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│   Controller     │  ← Route handling
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│    Service       │  ← Business logic
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│     Prisma       │  ← Database access
└──────────────────┘
```

## Multi-tenancy

MonkAPI supporte le multi-tenant via `merchant_id`:

```typescript
// Guard merchant
@Injectable()
export class MerchantGuard implements CanActivate {
  canActivate(context: ExecutionContext): boolean {
    const request = context.switchToHttp().getRequest();
    const merchantId = request.headers['x-merchant-id'];

    if (!merchantId) {
      throw new UnauthorizedException('Merchant ID required');
    }

    request.merchantId = parseInt(merchantId);
    return true;
  }
}

// Utilisation dans les services
@Injectable()
export class BookingService {
  async findAll(merchantId: number) {
    return this.prisma.booking.findMany({
      where: { merchant_id: merchantId }
    });
  }
}
```

### Isolation des donnees

```sql
-- Chaque requete est filtree par merchant_id
SELECT * FROM bookings WHERE merchant_id = ? AND ...
SELECT * FROM users WHERE merchant_id = ? AND ...
SELECT * FROM drivers WHERE merchant_id = ? AND ...
```

## Authentification

### Flow JWT

```
┌──────────┐      ┌──────────┐      ┌──────────┐
│  Client  │      │   API    │      │    DB    │
└────┬─────┘      └────┬─────┘      └────┬─────┘
     │                 │                 │
     │  POST /login    │                 │
     │────────────────>│                 │
     │                 │  Verify user    │
     │                 │────────────────>│
     │                 │<────────────────│
     │                 │                 │
     │  Access Token   │                 │
     │  Refresh Token  │                 │
     │<────────────────│                 │
     │                 │                 │
     │  GET /profile   │                 │
     │  + Bearer Token │                 │
     │────────────────>│                 │
     │                 │  Validate JWT   │
     │                 │────────────────>│
     │                 │<────────────────│
     │  User Data      │                 │
     │<────────────────│                 │
```

### Types de tokens

| Token | Duree | Usage |
|-------|-------|-------|
| Access Token | 7 jours | Requetes API |
| Refresh Token | 30 jours | Renouveler access token |
| OTP | 5 minutes | Verification |

## Cache Strategy

### Niveaux de cache

```
┌─────────────────────────────────────────────────────┐
│                    Application                       │
├─────────────────────────────────────────────────────┤
│  L1: In-Memory Cache (Node.js)                      │
│  - TTL: 60s                                          │
│  - Donnees frequentes                                │
├─────────────────────────────────────────────────────┤
│  L2: Redis Cache                                     │
│  - TTL: 5-60 minutes                                 │
│  - Sessions, configurations                          │
├─────────────────────────────────────────────────────┤
│  L3: MySQL                                           │
│  - Source de verite                                  │
└─────────────────────────────────────────────────────┘
```

### Patterns de cache

```typescript
// Cache-aside pattern
async getVehicleTypes(merchantId: number) {
  const cacheKey = `vehicle_types:${merchantId}`;

  // 1. Check cache
  let types = await this.cache.get(cacheKey);

  if (!types) {
    // 2. Load from DB
    types = await this.prisma.vehicleType.findMany({
      where: { merchant_id: merchantId, is_active: true }
    });

    // 3. Store in cache
    await this.cache.set(cacheKey, types, 300); // 5 min
  }

  return types;
}
```

## Event-Driven Architecture

### Events systeme

```typescript
// Emission d'evenement
this.eventEmitter.emit('booking.created', {
  bookingId: booking.id,
  userId: booking.user_id,
  merchantId: booking.merchant_id,
});

// Handler d'evenement
@OnEvent('booking.created')
async handleBookingCreated(payload: BookingCreatedEvent) {
  // Notifier le user
  await this.notificationService.send(payload.userId, {
    title: 'Reservation confirmee',
    body: `Votre course #${payload.bookingId} est confirmee`,
  });

  // Demarrer la recherche de chauffeur
  await this.driverSearchQueue.add('search', payload);
}
```

### Queues BullMQ

```
┌─────────────────────────────────────────────────────┐
│                     Producers                        │
│  ┌─────────┐  ┌─────────┐  ┌─────────┐             │
│  │Booking  │  │ Driver  │  │Payment  │             │
│  │Service  │  │ Service │  │Service  │             │
│  └────┬────┘  └────┬────┘  └────┬────┘             │
└───────┼────────────┼────────────┼───────────────────┘
        │            │            │
        ▼            ▼            ▼
┌─────────────────────────────────────────────────────┐
│                    Redis Queues                      │
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐ │
│  │booking-queue│  │driver-queue │  │notify-queue │ │
│  └─────────────┘  └─────────────┘  └─────────────┘ │
└───────┼────────────────┼────────────────┼───────────┘
        │                │                │
        ▼                ▼                ▼
┌─────────────────────────────────────────────────────┐
│                     Workers                          │
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐ │
│  │  Assign     │  │  Location   │  │    Push     │ │
│  │  Driver     │  │  Update     │  │Notification │ │
│  └─────────────┘  └─────────────┘  └─────────────┘ │
└─────────────────────────────────────────────────────┘
```

## WebSocket Architecture

### Namespaces Socket.io

```typescript
// Gateway principal
@WebSocketGateway({
  namespace: '/booking',
  cors: { origin: '*' }
})
export class BookingGateway {

  @SubscribeMessage('track')
  handleTrack(client: Socket, bookingId: string) {
    client.join(`booking:${bookingId}`);
  }

  // Broadcast position chauffeur
  broadcastDriverLocation(bookingId: number, location: Location) {
    this.server.to(`booking:${bookingId}`).emit('driver:location', location);
  }
}
```

### Rooms et events

```
┌─────────────────────────────────────────────────────┐
│                   Socket.io Server                   │
├─────────────────────────────────────────────────────┤
│  Namespace: /booking                                 │
│  ├── Room: booking:123                               │
│  │   ├── User (socket_abc)                           │
│  │   └── Driver (socket_xyz)                         │
│  ├── Room: booking:456                               │
│  │   └── User (socket_def)                           │
│  └── ...                                             │
├─────────────────────────────────────────────────────┤
│  Namespace: /driver                                  │
│  ├── Room: merchant:1                                │
│  │   ├── Driver #1                                   │
│  │   ├── Driver #2                                   │
│  │   └── Driver #3                                   │
│  └── ...                                             │
├─────────────────────────────────────────────────────┤
│  Namespace: /chat                                    │
│  ├── Room: conversation:789                          │
│  │   ├── User                                        │
│  │   └── Driver                                      │
│  └── ...                                             │
└─────────────────────────────────────────────────────┘
```

## Base de donnees

### Schema simplifie

```
┌─────────────┐     ┌─────────────┐     ┌─────────────┐
│  Merchant   │────<│    User     │────<│   Booking   │
└─────────────┘     └─────────────┘     └──────┬──────┘
       │                                        │
       │            ┌─────────────┐            │
       └───────────>│   Driver    │>───────────┘
                    └─────────────┘
                           │
                    ┌──────┴──────┐
                    │   Vehicle   │
                    └─────────────┘
```

### Relations principales

| Parent | Enfant | Relation |
|--------|--------|----------|
| Merchant | User | 1:N |
| Merchant | Driver | 1:N |
| Merchant | Booking | 1:N |
| User | Booking | 1:N |
| Driver | Booking | 1:N |
| Driver | Vehicle | 1:1 |
| Booking | Payment | 1:1 |
| Booking | Rating | 1:1 |

## Securite

### Mesures implementees

| Mesure | Implementation |
|--------|----------------|
| Authentication | JWT + Refresh tokens |
| Authorization | Guards NestJS |
| Rate Limiting | NGINX + @nestjs/throttler |
| Input Validation | class-validator |
| SQL Injection | Prisma ORM (parametrized) |
| XSS | Helmet middleware |
| CORS | Configuration stricte |
| HTTPS | TLS via Ingress |
| Secrets | Kubernetes Secrets |

### Headers de securite

```typescript
// Helmet configuration
app.use(helmet({
  contentSecurityPolicy: true,
  crossOriginEmbedderPolicy: true,
  crossOriginOpenerPolicy: true,
  crossOriginResourcePolicy: true,
  dnsPrefetchControl: true,
  frameguard: true,
  hidePoweredBy: true,
  hsts: true,
  ieNoOpen: true,
  noSniff: true,
  originAgentCluster: true,
  permittedCrossDomainPolicies: true,
  referrerPolicy: true,
  xssFilter: true,
}));
```

## Monitoring

### Metriques exposees

```
GET /api/health/metrics

# Metriques Prometheus
http_requests_total{method="GET",path="/api/v1/bookings",status="200"} 1234
http_request_duration_seconds{method="GET",path="/api/v1/bookings"} 0.045
database_connections_active 5
redis_connections_active 2
queue_jobs_waiting{queue="notifications"} 12
queue_jobs_completed{queue="notifications"} 5678
```

### Health checks

| Endpoint | Description |
|----------|-------------|
| `/api/health` | Check basique |
| `/api/health/live` | Liveness probe |
| `/api/health/ready` | Readiness probe |
| `/api/health/metrics` | Metriques Prometheus |

## Scalabilite

### Horizontal Pod Autoscaler

```yaml
spec:
  minReplicas: 3
  maxReplicas: 20
  metrics:
    - type: Resource
      resource:
        name: cpu
        target:
          type: Utilization
          averageUtilization: 70
```

### Considerations

1. **Stateless** - Pas d'etat en memoire entre requetes
2. **Sessions** - Stockees dans Redis
3. **Uploads** - Stockes sur S3
4. **WebSockets** - Redis adapter pour sync entre pods
5. **Jobs** - BullMQ avec Redis pour distribution
